/*
 * Java
 *
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http;

import static ej.hoka.http.HttpConstants.ALLOW;
import static ej.hoka.http.HttpConstants.CLOSE;
import static ej.hoka.http.HttpConstants.EMPTY;
import static ej.hoka.http.HttpConstants.HEADER_ACCEPT_ENCODING;
import static ej.hoka.http.HttpConstants.HEADER_CONNECTION;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_BADREQUEST;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_INTERNALERROR;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_METHODNOTALLOWED;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_NOTACCEPTABLE;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_NOTFOUND;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_REQUESTTIMEOUT;
import static ej.hoka.http.HttpConstants.KEEP_ALIVE;
import static ej.hoka.http.HttpConstants.TAB;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.Date;
import java.util.List;

import javax.net.ServerSocketFactory;

import ej.hoka.http.encoding.ContentEncoding;
import ej.hoka.http.encoding.EncodingRegistry;
import ej.hoka.http.encoding.HttpUnsupportedEncodingException;
import ej.hoka.http.requesthandler.RequestHandler;
import ej.hoka.http.requesthandler.StaticFilesHandler;
import ej.hoka.http.support.Mime;
import ej.hoka.log.HokaLogger;
import ej.hoka.tcp.TcpServer;

/**
 * HTTP Server.
 * <p>
 * <b>Features + limitations: </b>
 * <ul>
 * <li>CLDC 1.1</li>
 * <li>No fixed configuration files, logging, authorization, encryption.</li>
 * <li>Supports parameter parsing of GET and POST methods</li>
 * <li>Supports both dynamic content and file serving</li>
 * <li>Never caches anything</li>
 * <li>Doesn't limit bandwidth, request time or simultaneous connections</li>
 * <li>Contains a built-in list of most common MIME types</li>
 * <li>All header names are converted to lower case</li>
 * <li>Keep-alive is not supported</li>
 * </ul>
 * <p>
 * <b>Example:</b>
 *
 * <pre>
 * // get a new server which uses a ResourceRequestHandler
 * HttpServer http = HttpServer.builder() //
 * 		.port(8080) //
 * 		.simultaneousConnections(3) //
 * 		.workerCount(3) //
 * 		.build();
 *
 * // register a route
 * http.get("/", requestHandler);
 *
 * // start the server
 * http.start();
 * </pre>
 * <p>
 * To parse the request and write the response, a buffer size of 4096 by default is used. To change this value, set the
 * property "hoka.buffer.size".
 *
 */
public class HttpServer {

	/**
	 * The HTML line break tag.
	 */
	private static final String HTML_BR = "<br/>"; //$NON-NLS-1$

	/**
	 * The underlying TCP server.
	 */
	private final TcpServer server;

	/**
	 * Number of jobs per sessions.
	 */
	private final int sessionJobsCount;

	private final RouteHandler routesHandler;

	private final EncodingRegistry encodingRegistry;

	/**
	 * not found exception message and mime type
	 */
	private String notFoundErrorMessage = EMPTY;
	private String notFoundErrorContentType = Mime.MIME_PLAINTEXT;

	/**
	 * internal server exception message and mime type
	 */
	private String internalServerErrorMessage = EMPTY;
	private String internalServerErrorContentType = Mime.MIME_PLAINTEXT;

	/**
	 * Array of {@link Thread} for the session jobs.
	 */
	private Thread[] jobs;

	/**
	 * Activate development mode to send Java exception stack trace to the client.
	 */
	private final boolean devMode;

	/**
	 * Activate strict encoding support. RFC2616 14.3
	 */
	private final boolean strictAcceptEncoding;

	/**
	 * Configure and build an {@link HttpServer} instance.
	 *
	 * @return {@link HttpServerBuilder} instance
	 */
	public static HttpServerBuilder builder() {
		return new HttpServerBuilder();
	}

	/**
	 * {@link HttpServerBuilder} builder
	 */
	public static class HttpServerBuilder {

		private int port = 0;
		private int simultaneousConnections = 1;
		private int workerCount = 1;
		private String apiBase = EMPTY;
		private ServerSocketFactory ssf = ServerSocketFactory.getDefault();
		private int connectionTimeout = 60000; // 60s
		private EncodingRegistry encodingRegistry = new EncodingRegistry();
		private StaticFilesHandler staticFilesHandler = null;
		private boolean devMode = false;
		private boolean strictAcceptEncoding = false;
		private boolean trailingSlashSupport = false;

		private HttpServerBuilder() {
			// no-op
		}

		/**
		 * Set the server port number.
		 *
		 * @param port
		 *            server port. use 0 set setup a random available port.
		 *
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if port is {@code < 0}
		 */
		public HttpServerBuilder port(int port) throws IllegalArgumentException {
			if (port < 0) {
				throw new IllegalArgumentException();
			}
			this.port = port;
			return this;
		}

		/**
		 * set the max simultaneous connections
		 *
		 * @param simultaneousConnections
		 *            max simultaneous connections that can be handled by the server.
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             - if simultaneousConnections is {@code <= 0};
		 *
		 */
		public HttpServerBuilder simultaneousConnections(int simultaneousConnections) throws IllegalArgumentException {
			if (this.simultaneousConnections <= 0) {
				throw new IllegalArgumentException();
			}
			this.simultaneousConnections = simultaneousConnections;
			return this;
		}

		/**
		 * set threads count that handles incoming request
		 *
		 * @param workerCount
		 *            number of threads to handle incoming requests
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             - if workerCount is {@code <= 0};
		 */
		public HttpServerBuilder workerCount(int workerCount) throws IllegalArgumentException {
			if (workerCount <= 0) {
				throw new IllegalArgumentException();
			}
			this.workerCount = workerCount;
			return this;
		}

		/**
		 *
		 * Add a path common base that will be appended to all routes mapped without starting with a forward slash '/'.
		 *
		 * This is useful when dealing with rest API to set the rest api URI base once.
		 *
		 * The base should start with a '/' and also ends with a '/'.
		 *
		 * @param apiBase
		 *            the api base path
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if apiBase is null or empty
		 */
		public HttpServerBuilder apiBase(String apiBase) throws IllegalArgumentException {
			if (apiBase == null || apiBase.isEmpty()) {
				throw new IllegalArgumentException();
			}

			this.apiBase = apiBase;
			return this;
		}

		/**
		 * Set Server Socket Factory. this used to set ssl
		 *
		 * @param ssf
		 *            Server socket factory to setup HTTPS
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if ServerSocketFactory parameter is null
		 *
		 */
		public HttpServerBuilder secure(ServerSocketFactory ssf) throws IllegalArgumentException {
			if (ssf == null) {
				throw new IllegalArgumentException();
			}
			this.ssf = ssf;
			return this;
		}

		/**
		 * By default, server is configured to keep connection open during one minute if possible.
		 *
		 * A timeout of zero is interpreted as an infinite timeout.
		 *
		 * @param connectionTimeout
		 *            the timeout duration.
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if connectionTimeout {@code < 0 }
		 */
		public HttpServerBuilder connectionTimeout(int connectionTimeout) throws IllegalArgumentException {
			if (connectionTimeout < 0) {
				throw new IllegalArgumentException();
			}
			this.connectionTimeout = connectionTimeout;
			return this;
		}

		/**
		 * If this method is not called an instance of {@link EncodingRegistry} is used by the server.
		 *
		 * @param encodingRegistry
		 *            the registry of available encoding handlers.
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if encodingRegistry is null
		 */
		public HttpServerBuilder encodingRegistry(EncodingRegistry encodingRegistry) throws IllegalArgumentException {
			if (encodingRegistry == null) {
				throw new IllegalArgumentException();
			}
			this.encodingRegistry = encodingRegistry;
			return this;
		}

		/**
		 * RFC2616 14.3
		 *
		 * Activate strict content encoding acceptance.
		 *
		 * The server will returns a 406 (Not Acceptable) for every request with a header Accept-Encoding with no
		 * encoding supported by the server.
		 *
		 * Register custom content encoding using {@link EncodingRegistry#registerContentEncoding(ContentEncoding)}
		 *
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 */
		public HttpServerBuilder withStrictAcceptContentEncoding() {
			this.strictAcceptEncoding = true;
			return this;
		}

		/**
		 *
		 * Activate trailing slash matching support in URI.
		 *
		 * When used URI '/hello' and '/hello/' will point to different request handler
		 *
		 * otherwise URI '/hello' and '/hello/' are the same
		 *
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 */
		public HttpServerBuilder withTrailingSlashSupport() {
			this.trailingSlashSupport = true;
			return this;
		}

		/**
		 * Set the static files handler to serve static files.
		 *
		 * Use this only for public resources.
		 *
		 * @param handler
		 *            the {@link StaticFilesHandler} instance
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 * @throws IllegalArgumentException
		 *             if the handler is null
		 */
		public HttpServerBuilder staticFilesHandler(StaticFilesHandler handler) throws IllegalArgumentException {
			if (handler == null) {
				throw new IllegalArgumentException();
			}
			this.staticFilesHandler = handler;
			return this;
		}

		/**
		 * Activate development mode. The server will send stack trace of thrown exceptions to the clients.
		 *
		 * @return {@link HttpServerBuilder} to continue setting up the server parameters
		 */
		public HttpServerBuilder developmentMode() {
			this.devMode = true;
			return this;
		}

		/**
		 *
		 * @return an instance of {@link HttpServer} with the configured parameters.
		 */
		public HttpServer build() {
			return new HttpServer(this.port, this.simultaneousConnections, this.workerCount, this.ssf,
					this.connectionTimeout, this.encodingRegistry, this.apiBase, this.staticFilesHandler,
					this.strictAcceptEncoding, this.trailingSlashSupport, this.devMode);
		}
	}

	/**
	 * Constructs a HTTP server that manage jobs to handle the connections from <code>tcpServer</code> with
	 * <code>requestHandler</code>.
	 *
	 * @param staticFilesHandler2
	 *
	 */
	private HttpServer(int port, int maxSimultaneousConnection, final int jobCount,
			ServerSocketFactory serverSocketFactory, int connectionTimeout, final EncodingRegistry encodingRegistry,
			String apiBase, StaticFilesHandler staticFilesHandler, boolean strictAcceptEncoding,
			boolean trailingSlashSupport, boolean devMode) {

		this.server = new TcpServer(port, maxSimultaneousConnection, serverSocketFactory, connectionTimeout);
		this.sessionJobsCount = jobCount;
		this.encodingRegistry = encodingRegistry;
		this.routesHandler = new RouteHandler(apiBase, staticFilesHandler, trailingSlashSupport);
		this.strictAcceptEncoding = strictAcceptEncoding;
		this.devMode = devMode;
	}

	/**
	 * Start the {@link HttpServer} (in a dedicated thread): start listening for connections and start jobs to process
	 * opened connections.
	 * <p>
	 * Multiple start is not allowed.
	 *
	 * @throws IOException
	 *             if an error occurs during the creation of the socket.
	 */
	public void start() throws IOException {
		this.server.start();

		this.jobs = new Thread[this.sessionJobsCount];

		for (int i = this.sessionJobsCount - 1; i >= 0; i--) {
			Thread job = new Thread(newJob(), "HTTP-JOB-" + i); //$NON-NLS-1$
			this.jobs[i] = job;
			job.start();
		}
	}

	/**
	 * Stops the {@link HttpServer}. Stops listening for connections. This method blocks until all session jobs are
	 * stopped.
	 */
	public void stop() {
		this.server.stop();

		for (int i = this.jobs.length - 1; i >= 0; i--) {
			try {
				this.jobs[i].join();
			} catch (InterruptedException e) {
				// nothing to do on interrupted exception
			}
		}
	}

	/**
	 * Returns a new job process as {@link Runnable}.
	 *
	 * @return a new job process as {@link Runnable}.
	 */
	private Runnable newJob() {
		return new Runnable() {
			@Override
			public void run() {
				while (true) {
					try (Socket connection = HttpServer.this.server.getNextStreamConnection()) {
						if (connection == null) {
							// server stopped
							return;
						}

						HokaLogger.instance.trace(connection.hashCode() + TAB
								+ connection.getInetAddress().getHostAddress() + "\t processing"); //$NON-NLS-1$

						handleConnection(connection);

						HokaLogger.instance.trace(connection.hashCode() + TAB
								+ connection.getInetAddress().getHostAddress() + "\t closed"); //$NON-NLS-1$

					} catch (IOException e) {
						HokaLogger.instance.error(e);
					}
				}
			}
		};
	}

	private void handleConnection(Socket connection) {
		try (InputStream inputStream = new BufferedInputStream(connection.getInputStream(),
				Config.getInstance().getBufferSize()); OutputStream outputStream = connection.getOutputStream()) {
			boolean keepAlive;
			do {
				HttpRequest request = null;
				HttpResponse response = new HttpResponse();
				ContentEncoding encodingHandler = null;

				try {

					request = new HttpRequest(inputStream, this.encodingRegistry);

					// RFC2616 14.3
					if (this.strictAcceptEncoding) {
						String accept = request.getHeader(HEADER_ACCEPT_ENCODING);
						encodingHandler = this.encodingRegistry.getAcceptEncodingHandler(accept);
						if (this.encodingRegistry.getAcceptEncodingHandler(accept) == null) {
							throw new HttpUnsupportedEncodingException(HTTP_STATUS_NOTACCEPTABLE, accept);
						}
					}

					this.routesHandler.process(request, response);

					final String requestConnectionHeader = request.getHeader(HEADER_CONNECTION);
					final String responseConnectionHeader = response.getHeader(HEADER_CONNECTION);
					keepAlive = KEEP_ALIVE.equalsIgnoreCase(requestConnectionHeader)
							&& !CLOSE.equalsIgnoreCase(responseConnectionHeader);

				} catch (final HaltException e) {
					response.setStatus(e.getStatus());
					response.setData(e.getBody());
				} catch (MethodNotAllowedException e) {
					handleMethodNotAllowedError(response, e); // 405
				} catch (final RouteNotFoundException e) {
					handleNotFoundError(response); // 404
				} catch (final IllegalArgumentException e) {
					handleError(response, HTTP_STATUS_BADREQUEST, e);
				} catch (final HttpUnsupportedEncodingException e) {
					handleError(response, HTTP_STATUS_NOTACCEPTABLE, e);
				} catch (final SocketTimeoutException e) {
					handleError(response, HTTP_STATUS_REQUESTTIMEOUT, e);
				} catch (final IOException e) {
					throw e;
				} catch (final Throwable e) {
					handleInternalServerError(response, e);
				} finally {
					// keepAlive = isKeepAlive(request);
					keepAlive = false; // TODO : not supported
				}

				if (keepAlive) {
					response.addHeader(HEADER_CONNECTION, KEEP_ALIVE);
				} else {
					response.addHeader(HEADER_CONNECTION, CLOSE);
				}

				// log the response
				String msg = createResponseLogMsg(request == null ? EMPTY : request.getURI(), response.getStatus(),
						connection.hashCode(), connection.getInetAddress().getHostAddress());
				HokaLogger.instance.debug(msg);

				// send response back
				response.sendResponse(outputStream, encodingHandler, this.encodingRegistry,
						Config.getInstance().getBufferSize());

			} while (keepAlive);
		} catch (final IOException e) {
			// connection lost
			HokaLogger.instance.error(connection.hashCode() + TAB + connection.getInetAddress().toString(), e);
		}
	}

	private String createResponseLogMsg(String uri, String status, int connection, String address) {
		StringBuilder msg = new StringBuilder();
		msg.append(connection);
		msg.append(TAB);
		msg.append(address);
		msg.append(TAB);
		msg.append(new Date());
		msg.append(TAB);
		msg.append(uri);
		msg.append(TAB);
		msg.append(status);
		return msg.toString();
	}

	/**
	 * Set response to method not allowed with allow header
	 *
	 * @param response
	 *            http response
	 * @param e
	 *            MethodNotAllowedException
	 */
	private void handleMethodNotAllowedError(HttpResponse response, MethodNotAllowedException e) {
		response.setStatus(HTTP_STATUS_METHODNOTALLOWED);
		response.setData(EMPTY);
		response.addHeader(ALLOW, e.getAllowHeader());
	}

	/**
	 * @param response
	 * @param e
	 */
	private void handleInternalServerError(final HttpResponse response, final Throwable e) {
		response.setStatus(HTTP_STATUS_INTERNALERROR);
		if (this.devMode) {
			response.setMimeType(Mime.MIME_HTML);
			response.setData(createHtmlError(HTTP_STATUS_INTERNALERROR, e.getMessage(), e));
		} else {
			response.setMimeType(this.internalServerErrorContentType);
			response.setData(this.internalServerErrorMessage);
		}
	}

	/**
	 * @param request
	 *            http request
	 * @return return true if the request has the keep alive header, false otherwise
	 */
	private boolean isKeepAlive(HttpRequest request) {
		return request != null && request.getHeader(HEADER_CONNECTION).equalsIgnoreCase(KEEP_ALIVE);
	}

	private void handleError(final HttpResponse response, final String status, final Exception e) {
		response.setStatus(status);
		if (this.devMode) {
			response.setMimeType(Mime.MIME_HTML);
			response.setData(createHtmlError(status, e.getMessage(), e));
		} else {
			response.setData(EMPTY);
		}
	}

	private void handleNotFoundError(final HttpResponse response) {
		response.setStatus(HTTP_STATUS_NOTFOUND);
		response.setMimeType(this.notFoundErrorContentType);
		if (this.notFoundErrorMessage == null || this.notFoundErrorMessage.isEmpty()) {
			response.setData(EMPTY);
		} else {
			response.setData(this.notFoundErrorMessage);
		}
	}

	/**
	 * Creates a HTML representation of the stack trace of <code>t</code>.
	 * <p>
	 * Only the relevant part of the stack trace is dumped. The exception superclasses constructors and the job internal
	 * calls are skipped.
	 *
	 * @param t
	 *            the throwable to dump.
	 * @return the HTML representation of the stack trace as a {@link String}.
	 */
	private static String getHtmlExceptionStackTrace(Throwable t) {
		StringBuilder fullMessageBuilder = new StringBuilder();

		String message = t.getMessage();
		if (message != null) {
			fullMessageBuilder.append(message);
			fullMessageBuilder.append(HTML_BR);
		}

		StackTraceElement[] stackTrace = t.getStackTrace();

		int i = 0;
		String className;

		// Skip all the exception superclasses constructors
		className = "java.lang"; //$NON-NLS-1$
		while (i < stackTrace.length && stackTrace[i].getClassName().startsWith(className)) {
			i++;
		}

		// Append only the stack trace up to this class call to RequestHandler#process.
		if (i > 0 && i - 1 < stackTrace.length) {
			fullMessageBuilder.append(stackTrace[i - 1].toString()).append(HTML_BR);
		}

		className = HttpServer.class.getName();
		final String prefix = "&nbsp;&nbsp;&nbsp;&nbsp;at "; //$NON-NLS-1$
		for (; i < stackTrace.length && !stackTrace[i].getClassName().equals(className); i++) {
			fullMessageBuilder.append(prefix).append(stackTrace[i].toString()).append(HTML_BR);
		}

		return fullMessageBuilder.toString();
	}

	/**
	 * Create a {@link HttpResponse} to write the <code>msg</code> for the given <code>status</code>.
	 *
	 * @param status
	 *            the error status. One of <code>HTTP_STATUS_*</code> constant of the {@link HttpConstants} interface.
	 * @param msg
	 *            an optional error message to add in response.
	 * @return a {@link HttpResponse} that represent the error.
	 * @see HttpConstants#HTTP_STATUS_BADREQUEST
	 * @see HttpConstants#HTTP_STATUS_FORBIDDEN
	 * @see HttpConstants#HTTP_STATUS_INTERNALERROR
	 * @see HttpConstants#HTTP_STATUS_MEDIA_TYPE
	 * @see HttpConstants#HTTP_STATUS_METHODNOTALLOWED
	 * @see HttpConstants#HTTP_STATUS_NOTACCEPTABLE
	 * @see HttpConstants#HTTP_STATUS_NOTFOUND
	 * @see HttpConstants#HTTP_STATUS_NOTIMPLEMENTED
	 * @see HttpConstants#HTTP_STATUS_NOTMODIFIED
	 * @see HttpConstants#HTTP_STATUS_OK
	 * @see HttpConstants#HTTP_STATUS_REDIRECT
	 */
	private static String createHtmlError(String status, String msg, Throwable t) {
		StringBuilder buffer = new StringBuilder();
		buffer.append("<html><head><title>"); //$NON-NLS-1$
		buffer.append(status);
		buffer.append("</title></head><body>"); //$NON-NLS-1$
		buffer.append("<h3>").append(status).append("</h3>"); //$NON-NLS-1$ //$NON-NLS-2$
		if (msg != null && !msg.isEmpty()) {
			buffer.append("<p>").append(msg).append("</p>"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (t != null) {
			buffer.append("<hr/>"); //$NON-NLS-1$
			buffer.append("<b>Exception:</b>"); //$NON-NLS-1$
			buffer.append("<p>").append(getHtmlExceptionStackTrace(t)).append("</p>"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		buffer.append("</body></html>"); //$NON-NLS-1$
		return buffer.toString();
	}

	/**
	 * Map the handler to HTTP GET requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void get(final String path, final RequestHandler handler) {
		get(path, null, handler);
	}

	/**
	 * Map the handler to HTTP GET requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void get(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.GET, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP POST requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void post(final String path, final RequestHandler handler) {
		post(path, null, handler);
	}

	/**
	 * Map the handler to HTTP POST requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void post(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.POST, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP PUT requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void put(final String path, final RequestHandler handler) {
		put(path, null, handler);
	}

	/**
	 * Map the handler to HTTP PUT requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void put(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.PUT, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP POST requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void delete(final String path, final RequestHandler handler) {
		delete(path, null, handler);
	}

	/**
	 * Map the handler to HTTP DELETE requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void delete(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.DELETE, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP HEAD requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void head(final String path, final RequestHandler handler) {
		head(path, null, handler);
	}

	/**
	 * Map the handler to HTTP HEAD requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void head(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.HEAD, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP CONNECT requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void connect(final String path, final RequestHandler handler) {
		connect(path, null, handler);
	}

	/**
	 * Map the handler to HTTP CONNECT requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void connect(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.CONNECT, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP OPTIONS requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void options(final String path, final RequestHandler handler) {
		options(path, null, handler);
	}

	/**
	 * Map the handler to HTTP OPTIONS requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void options(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.OPTIONS, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP TRACE requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void trace(final String path, final RequestHandler handler) {
		trace(path, null, handler);
	}

	/**
	 * Map the handler to HTTP TRACE requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void trace(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.TRACE, path, acceptType, handler);
	}

	/**
	 * Map the handler to HTTP PATCH requests on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void patch(final String path, final RequestHandler handler) {
		patch(path, null, handler);
	}

	/**
	 * Map the handler to HTTP PATCH requests on the path
	 *
	 * @param path
	 *            request path
	 * @param acceptType
	 *            accepted type
	 * @param handler
	 *            request handler
	 */
	public void patch(final String path, final String acceptType, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.PATCH, path, acceptType, handler);
	}

	/**
	 * Map the handler as a before filter on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void before(String path, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.BEFORE, path, handler);
	}

	/**
	 * Map the handler as a before filter on all path
	 *
	 * @param handler
	 *            request handler
	 */
	public void before(final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.BEFORE_ALL, RouteHandler.ALL_PATH, handler);
	}

	/**
	 * Map the handler as an after filter on the path
	 *
	 * @param path
	 *            request path
	 * @param handler
	 *            request handler
	 */
	public void after(String path, final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.AFTER, path, handler);
	}

	/**
	 * Map the handler as an after filter on all paths
	 *
	 * @param handler
	 *            request handler
	 */
	public void after(final RequestHandler handler) {
		this.routesHandler.add(HttpRequest.AFTER_ALL, RouteHandler.ALL_PATH, handler);
	}

	/**
	 * Immediately Stop the execution and return a 200 OK HTTP response.
	 */
	public static final void halt() {
		throw new HaltException();
	}

	/**
	 * immediately Stop the execution and return an HTTP response with the status.
	 *
	 * @param status
	 *            http status
	 */
	public static final void halt(String status) {
		throw new HaltException(status);
	}

	/**
	 * immediately Stop the execution and return an HTTP response with the status and body.
	 *
	 * @param status
	 *            http status
	 * @param body
	 *            response body
	 */
	public static final void halt(String status, String body) {
		throw new HaltException(status, body);
	}

	/**
	 * Override not found error 404
	 *
	 * @param response
	 *            error message returned as text/plain
	 */
	public final void notFoundError(final String response) {
		if (response == null) {
			throw new IllegalArgumentException();
		}
		notFoundError(response, Mime.MIME_PLAINTEXT);
	}

	/**
	 * Override not found error 404.
	 *
	 * @param response
	 *            error message
	 * @param contentType
	 *            response content type. application/json for example
	 */
	public final void notFoundError(final String response, final String contentType) {
		if (response == null || contentType == null) {
			throw new IllegalArgumentException();
		}
		this.notFoundErrorMessage = response;
		this.notFoundErrorContentType = contentType;
	}

	/**
	 * Override internal server error message 501
	 *
	 * @param response
	 *            error message returned as text/plain
	 */
	public final void internalServerError(final String response) {
		if (response == null) {
			throw new IllegalArgumentException();
		}
		internalServerError(response, Mime.MIME_PLAINTEXT);
	}

	/**
	 * Override internal server error message 501
	 *
	 * @param response
	 *            error message
	 * @param contentType
	 *            response content type. application/json for example
	 */
	public final void internalServerError(final String response, final String contentType) {
		if (response == null || contentType == null) {
			throw new IllegalArgumentException();
		}
		this.internalServerErrorMessage = response;
		this.internalServerErrorContentType = contentType;
	}

	/**
	 * Map a handler to a specific exception
	 *
	 * @param exception
	 *            exception to handle
	 * @param handler
	 *            exception handler
	 */
	public final void exception(Class<? extends Exception> exception, RequestHandler handler) {
		this.routesHandler.addExceptionHandler(exception, handler);
	}

	/**
	 *
	 * @return list of registered routes
	 */
	public final List<Route> getRoutes() {
		return this.routesHandler.getRoutes();
	}

	/**
	 *
	 * @return the port number to which this server is listening or-1 if the socket is not bound yet.
	 */
	public int getPort() {
		return this.server.getPort();
	}

	/**
	 *
	 * @return the address to which this server is bound,or null if the socket is unbound.
	 */
	public InetAddress getInetAddress() {
		return this.server.getInetAddress();
	}

}
