/*
 * Java
 *
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http;

import java.util.Date;

/**
 * Response cookie
 */
public class Cookie {

	/**
	 * same site values
	 */
	public enum SameSite {
		Strict, Lax, None;
	}

	private final String name;
	private final String value;
	private String domain;
	private String path;
	private int maxAge;
	private Date expires;
	private boolean secured;
	private boolean httpOnly;
	private SameSite sameSite;

	Cookie(String name, String value, String domain, String path, int maxAge, Date expires, boolean secured,
			boolean httpOnly, SameSite sameSite) {
		this(name, value, maxAge);
		this.domain = domain;
		this.path = path;
		this.maxAge = maxAge;
		this.expires = expires;
		this.secured = secured;
		this.httpOnly = httpOnly;
		this.sameSite = sameSite;
	}

	Cookie(String name, String value, int maxAge) {
		this.name = name;
		this.value = value;
		this.maxAge = maxAge;
	}

	@Override
	public String toString() {
		final StringBuilder cookie = new StringBuilder();
		cookie.append(this.name).append("=").append(this.value); //$NON-NLS-1$

		if (this.maxAge != 0) {
			cookie.append("; Max-Age=").append(this.maxAge); //$NON-NLS-1$
		}

		if (this.expires != null) {
			cookie.append("; Expires=").append(this.expires.toString()); //$NON-NLS-1$
		}

		if (this.path != null && !this.path.isEmpty()) {
			cookie.append("; Path=").append(this.path); //$NON-NLS-1$
		}

		if (this.domain != null && !this.domain.isEmpty()) {
			cookie.append("; Domain=").append(this.domain); //$NON-NLS-1$
		}

		if (this.secured) {
			cookie.append("; Secure"); //$NON-NLS-1$
		}

		if (this.httpOnly) {
			cookie.append("; HttpOnly"); //$NON-NLS-1$
		}

		if (this.sameSite != null) {
			cookie.append("; SameSite=" + this.sameSite.name()); //$NON-NLS-1$
		}

		return cookie.toString();
	}

}
