/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.tool.ui.generator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;

/**
 * An implementation of this interface is required to convert a file in a format
 * not managed by MicroUI (custom ouput format different than MicroUI RAW
 * formats).
 * <p>
 * The generated file convert is not compatible with MicroUI API but it can be
 * used by some custom MicroUI extensions.
 */
public interface ImageConverter {

	/**
	 * Specifies the generated file's output type.
	 */
	enum OutputFileType {

		/**
		 * Specifies that the generated file is a binary resource. The file will be
		 * automatically added in the resources section.
		 */
		RESOURCE,

		/**
		 * Specifies that the generated file is an immutable file. The file will be
		 * automatically included during the application build.
		 */
		IMMUTABLE,

		/**
		 * Specifies that the generated file is another kind of resource. The converter
		 * has the responsability to embed this file with the application.
		 */
		OTHER,
	}

	/**
	 * Specifies the generated file: its relative path from the output folder and
	 * its type.
	 */
	interface OutputFile {

		/**
		 * Gets the relative path from the output folder.
		 *
		 * @return a relative path.
		 */
		String getRelativePath();

		/**
		 * Gets the file type.
		 *
		 * @return the file type.
		 */
		OutputFileType getType();
	}

	/**
	 * Generates a file from the given file in the given output folder.
	 * <p>
	 * This method allows to generate any types of file with any name.
	 * <p>
	 * By default, this method creates a file with the same relative path than input
	 * image plus the extension returned by {@link #getOutputFileExtension()}. Then
	 * it call {@link #generate(Image, OutputStream)}. This simplifies the
	 * implementation that just needs to convert the input file and fill the file
	 * output stream.
	 *
	 * @param image
	 *            the image to convert
	 * @param outputDir
	 *            the output folder
	 * @return an output file when a file has been generated, null otherwise.
	 * @throws IOException
	 */
	default OutputFile generate(Image image, File outputDir) throws IOException {
		final String relativePath = image.getRelativePath() + getOutputFileExtension();
		File file = new File(outputDir, relativePath);
		try (FileOutputStream fos = new FileOutputStream(file)) {
			if (generate(image, fos)) {
				fos.flush();
				return new OutputFile() {

					@Override
					public OutputFileType getType() {
						return OutputFileType.RESOURCE;
					}

					@Override
					public String getRelativePath() {
						return relativePath;
					}
				};
			}
		}

		// file has not been filled: delete it
		Files.delete(file.toPath());
		return null;
	}

	/**
	 * Converts the given image in the given output stream.
	 * <p>
	 * If this method returns true, the image is considered as encoded even if no
	 * byte has been written into the output stream. In this case the final file
	 * size is 0.
	 * <p>
	 * If this method returns false, the image is not considered as encoded even if
	 * some bytes have been written into the output stream. In this case the written
	 * bytes are lost.
	 *
	 * @param image
	 *            the image to convert
	 * @param stream
	 *            the output stream where write the encoded image.
	 * @return true when the image has been converted.
	 * @throws IOException
	 */
	boolean generate(Image image, OutputStream stream) throws IOException;

	/**
	 * Returns the file extension to apply as suffix for the converted images. At
	 * runtime, the converted image can be retrieved using the image classpath
	 * relative path (with its extension, for instance ".png") + this extension.
	 *
	 * @return the output file extension
	 */
	String getOutputFileExtension();

	/**
	 * Gets the exhaustive list of the list file extensions supported by this
	 * converter.
	 * <p>
	 * Only the files listed in a supported list file are given to this converter to
	 * be generated. The default extension (*.images.list) is implicitely added when
	 * no specific extension is returned (otherwise the default extension must be
	 * explicitely added).
	 *
	 * @return the list file extensions (with or without the first point '.') or an
	 *         empty array (null accepted)
	 */
	default String[] getListFileExtensions() {
		// no specific extension by default (only convert files listed in *.image.list
		// files)
		return new String[0];
	}
}
