/*******************************************************************************
 * Copyright (c) 2009, 2016 IBM Corp.
 * Copyright 2019-2021 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *    http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Dave Locke - initial API and implementation and/or initial documentation
 *    Ian Craggs - MQTT 3.1.1 support
 *    James Sutton - Automatic Reconnect & Offline Buffering
 *    MicroEJ Corp. - MicroPaho implementation and optimizations on MicroEJ
 */
package org.eclipse.paho.client.mqttv3;

import javax.net.SocketFactory;

import ej.annotation.Nullable;

/**
 * Holds the set of options that control how the client connects to a server.
 */
public class MqttConnectOptions {
	/**
	 * The default keep alive interval in seconds if one is not specified.
	 */
	public static final int KEEP_ALIVE_INTERVAL_DEFAULT = 60;
	/**
	 * The default connection timeout in seconds if one is not specified.
	 */
	public static final int CONNECTION_TIMEOUT_DEFAULT = 30;

	/**
	 * Mqtt Version 3.1.1.
	 */
	public static final int MQTT_VERSION_3_1_1 = 4;

	private int keepAliveInterval = KEEP_ALIVE_INTERVAL_DEFAULT;
	@Nullable
	private String userName;
	@Nullable
	private char[] password;
	@Nullable
	private SocketFactory socketFactory;
	private int connectionTimeout = CONNECTION_TIMEOUT_DEFAULT;

	/**
	 * Constructs a new <code>MqttConnectOptions</code> object using the default values.
	 *
	 * The defaults are:
	 * <ul>
	 * <li>The keepalive interval is 60 seconds</li>
	 * <li>The connection timeout period is 30 seconds</li>
	 * <li>A standard SocketFactory is used</li>
	 * </ul>
	 * More information about these values can be found in the setter methods.
	 */
	public MqttConnectOptions() {
		// Initialise Base MqttConnectOptions Object
	}

	/**
	 * Returns the password to use for the connection.
	 *
	 * @return the password to use for the connection, or <code>null</code> if no password is required for the
	 *         connection. <b>Important Note</b> : The returned char array is not cloned for memory efficiency reasons.
	 *         Be careful with it.
	 */
	@Nullable
	public char[] getPassword() {
		return this.password; // NOSONAR The returned char array is not cloned for memory efficiency reasons. Be careful
								// with it.
	}

	/**
	 * Sets the password to use for the connection.
	 *
	 * @param password
	 *            A Char Array of the password. <b>Important Note</b> : The char array is not cloned for memory
	 *            efficiency reasons. Be careful with it.
	 */
	public void setPassword(char[] password) {
		this.password = password; // NOSONAR The returned char array is not cloned for memory efficiency reasons.
									// Be careful with it.
	}

	/**
	 * Returns the user name to use for the connection.
	 *
	 * @return the user name to use for the connection, or <code>null</code> if no password is required for the
	 *         connection.
	 */
	@Nullable
	public String getUserName() {
		return this.userName;
	}

	/**
	 * Sets the user name to use for the connection.
	 *
	 * @param userName
	 *            The Username as a String
	 */
	public void setUserName(String userName) {
		this.userName = userName;
	}

	/**
	 * Returns the "keep alive" interval.
	 *
	 * @see #setKeepAliveInterval(int)
	 * @return the keep alive interval.
	 */
	public int getKeepAliveInterval() {
		return this.keepAliveInterval;
	}

	/**
	 * Returns the MQTT version.
	 *
	 * @return the MQTT version.
	 */
	public int getMqttVersion() {
		return MQTT_VERSION_3_1_1;
	}

	/**
	 * Sets the "keep alive" interval. This value, measured in seconds, defines the maximum time interval between
	 * messages sent or received. It enables the client to detect if the server is no longer available, without having
	 * to wait for the TCP/IP timeout. The client will ensure that at least one message travels across the network
	 * within each keep alive period. In the absence of a data-related message during the time period, the client sends
	 * a very small "ping" message, which the server will acknowledge. A value of 0 disables keepalive processing in the
	 * client.
	 * <p>
	 * The default value is 60 seconds
	 * </p>
	 *
	 * @param keepAliveInterval
	 *            the interval, measured in seconds, must be &gt;= 0.
	 * @throws IllegalArgumentException
	 *             if the keepAliveInterval was invalid
	 */
	public void setKeepAliveInterval(int keepAliveInterval) throws IllegalArgumentException {
		if (keepAliveInterval < 0) {
			throw new IllegalArgumentException();
		}
		this.keepAliveInterval = keepAliveInterval;
	}

	/**
	 * Returns the connection timeout value in seconds.
	 *
	 * @see #setConnectionTimeout(int)
	 * @return the connection timeout value in seconds.
	 */
	public int getConnectionTimeout() {
		return this.connectionTimeout;
	}

	/**
	 * Sets the connection timeout value. This value, measured in seconds, defines the maximum time interval the client
	 * will wait for the network connection to the MQTT server to be established. The default timeout is 30 seconds. A
	 * value of 0 disables timeout processing meaning the client will wait until the network connection is made
	 * successfully or fails.
	 *
	 * @param connectionTimeout
	 *            the timeout value, measured in seconds. It must be &gt;0;
	 */
	public void setConnectionTimeout(int connectionTimeout) {
		if (connectionTimeout < 0) {
			throw new IllegalArgumentException();
		}
		this.connectionTimeout = connectionTimeout;
	}

	/**
	 * Returns the socket factory that will be used when connecting, or <code>null</code> if one has not been set.
	 *
	 * @return The Socket Factory
	 */
	@Nullable
	public SocketFactory getSocketFactory() {
		return this.socketFactory;
	}

	/**
	 * Sets the <code>SocketFactory</code> to use. This allows an application to apply its own policies around the
	 * creation of network sockets. If using an SSL connection, an <code>SSLSocketFactory</code> can be used to supply
	 * application-specific security settings.
	 *
	 * @param socketFactory
	 *            the factory to use.
	 */
	public void setSocketFactory(SocketFactory socketFactory) {
		this.socketFactory = socketFactory;
	}

	/**
	 * Returns whether the client and server should remember state for the client across reconnects.
	 *
	 * @return the clean session flag
	 */
	public boolean isCleanSession() {
		return true;
	}

}
