/*
 * Java
 *
 * Copyright 2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.microej.tool.ui.generator;

import java.io.IOException;
import java.io.OutputStream;

/**
 * An implementation of this interface is required to convert an image is a
 * format not managed by MicroUI (custom ouput format different than MicroUI
 * Image RAW formats).
 * <p>
 * The resulted image is not compatible with MicroUI Image API but it can be
 * used by custom MicroUI extensions.
 */
public interface ImageConverter {

	/**
	 * Converts the given image in the given output stream.
	 * <p>
	 * If this method returns true, the image is considered as encoded even if no
	 * byte has been written into the output stream. In this case the final file
	 * size is 0.
	 * <p>
	 * If this method returns false, the image is not considered as encoded even if
	 * some bytes have been written into the output stream. In this case the written
	 * bytes are lost.
	 *
	 * @param image
	 *            the image to convert
	 * @param stream
	 *            the output stream where write the encoded image.
	 * @return true when the image has been converted.
	 * @throws IOException
	 */
	boolean generate(Image image, OutputStream stream) throws IOException;

	/**
	 * Returns the file extension to apply as suffix for the converted images. At
	 * runtime, the converted image can be retrieved using the image classpath
	 * relative path (with its extension, for instance ".png") + this extension.
	 *
	 * @return the output file extension
	 */
	String getOutputFileExtension();
}
