/*
 * Java
 *
 * Copyright 2009-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.benchmark;

/**
 * Represents a test to bench.
 * <p>
 * A test has three steps:
 * <ul>
 * <li>{@link #pre()}: Before launching the test, the test fields and
 * environment must be initialized during this step. This preparation time is
 * not accounted for.</li>
 * <li>{@link #run()}: The test itself.</li>
 * <li>{@link #post()}: After the test execution, this step is called to clean
 * the test environment.</li>
 * </ul>
 * The test (step {@link #run()}) can contain for any reasons some code which
 * must not be accounted for. In this case, this peace of code must be ran again
 * in the {@link #emptyRun()} in order to remove this time.
 * <p>
 * To have good results, the test must be executed several times. To determinate
 * the available number of iterations, a calibration step is performed during a
 * time defined by {@link #getCalibrationTime()}. In a second time, this number
 * of iterations is used to reach the bench time specified by
 * {@link #getRunTime()}.
 */
public abstract class BenchTest {

	/**
	 * Defines the default time (in milliseconds) to calibrate the number of
	 * iterations the test must be ran to reach the expected run time.
	 */
	protected static final long CALIBRATION_TIME = 500;

	/**
	 * Defines the test run time (in milliseconds)
	 */
	protected static final long RUN_TIME = 500;

	/**
	 * Gets the time to calibrate the number of iterations the test must be ran
	 * to reach the expected run time.
	 * <p>
	 * Default returned value is {@link #CALIBRATION_TIME}.
	 *
	 * @return a time in milliseconds
	 */
	protected long getCalibrationTime() {
		return CALIBRATION_TIME;
	}

	/**
	 * Gets the time the test must run to perform the bench.
	 * <p>
	 * Default returned value is {@link #RUN_TIME}.
	 *
	 * @return a time in milliseconds
	 */
	protected long getRunTime() {
		return RUN_TIME;
	}

	/**
	 * Gets the number of iterations the bench must be ran to reach the bench
	 * time specified by {@link #getRunTime()}.
	 * <p>
	 * This method can be overrided to force to use a specific number of
	 * iterations without taking in consideration {@link #getRunTime()}.
	 *
	 * @return a number of iterations (at least 1 iteration).
	 */
	public int getNumberOfIterations() {

		// get the bench test values (in case of bench changes its values
		// dynamically)
		long calibrationTime = getCalibrationTime();
		long runTime = getRunTime();

		// compute the number of iterations (run the indicator during
		// calibrationTime)
		long nbIterations = 0;
		long startCalibrationTime = System.currentTimeMillis();
		while (System.currentTimeMillis() - startCalibrationTime < calibrationTime) {
			++nbIterations;
			run();
		}

		return (int) (nbIterations == 0 ? 1 : (nbIterations * runTime) / calibrationTime);
	}

	/**
	 * Called to initialize bench test fields and environment before running the
	 * test.
	 * <p>
	 * By default there is noting to initialize.
	 */
	public void pre() {
		// nothing to do by default
	}

	/**
	 * Called to clean bench test environment after running the test.
	 * <p>
	 * By default there is noting to clean.
	 */
	public void post() {
		// nothing to do by default
	}

	/**
	 * Run the full test.
	 */
	public abstract void run();

	/**
	 * Called to remove from full test some time which must not be accounted for
	 * in bench results.
	 */
	public void emptyRun() {
		// nothing to do by default
	}

}
