/*
 * Java
 *
 * Copyright 2019-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.benchmark;

/**
 * Represents the results for a given test.
 */
public class BenchResult {

	/**
	 * The bench itself.
	 */
	public final BenchTest test;

	/**
	 * The number of iterations the bench has been ran to reach the time
	 * specified in {@link BenchTest#getRunTime()}.
	 */
	public final int iterations;

	/**
	 * The time (for all iterations, in nanoseconds) to remove from
	 * {@link #fullTime} in order to not accounted for extra work (see
	 * {@link BenchTest}).
	 */
	public final long emptyRunTime;

	/**
	 * The bench time (for all iterations, in nanoseconds).
	 */
	public final long fullTime;

	/**
	 * Creates a bench result for the given bench.
	 *
	 * @param test the bench which has produced the results
	 * @param fullTime the full bench time (in nanoseconds)
	 * @param emptyRunTime the bench time to remove from <code>fullTime</code>
	 * (in nanoseconds)
	 * @param iterations the number of iterations the bench has been ran.
	 *
	 */
	public BenchResult(BenchTest test, int iterations, long emptyRunTime, long fullTime) {
		this.test = test;
		this.iterations = iterations;
		this.emptyRunTime = emptyRunTime;
		this.fullTime = fullTime;
	}

	/**
	 * Gets the real bench time for one iteration removing the extra work time.
	 *
	 * @return the bench time in nanoseconds
	 */
	public long getBenchTime() {
		return (this.fullTime - this.emptyRunTime) / this.iterations;
	}
}
