/*
 * Copyright 2018 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.property;

import java.security.Permission;

import ej.annotation.Nullable;

/**
 * Permission to manipulate services using {@link PropertyLoader#getProperty(String)} or
 * {@link PropertyLoader#getProperty(String, String)} or {@link PropertyRegistry#setProperty(String, String)} or
 * {@link PropertyRegistry#removeProperty(String)}.
 */
public class PropertyPermission extends Permission {

	/**
	 * Action associated with {@link PropertyLoader#getProperty(String)} or
	 * {@link PropertyLoader#getProperty(String, String)}.
	 */
	public static final String GET_ACTION = "get"; //$NON-NLS-1$
	/**
	 * Action associated with {@link PropertyRegistry#setProperty(String, String)}.
	 */
	public static final String SET_ACTION = "set"; //$NON-NLS-1$
	/**
	 * Action associated with {@link PropertyRegistry#removeProperty(String)}.
	 */
	public static final String REMOVE_ACTION = "remove"; //$NON-NLS-1$

	private final String key;
	private final String action;

	/**
	 * Creates a permission for the given property.
	 * <p>
	 * The action can be one of <code>{@link #GET_ACTION}</code>, <code>{@link #SET_ACTION}</code> or
	 * <code>{@link #REMOVE_ACTION}</code>.
	 *
	 * @param parentName
	 *            the name of the service loader or service registry.
	 * @param key
	 *            the property key.
	 * @param action
	 *            the action realized with the service.
	 */
	public PropertyPermission(String parentName, String key, String action) {
		super(parentName + ':' + key);
		this.key = key;
		this.action = action;
	}

	@Override
	public String getActions() {
		return this.action;
	}

	/**
	 * Gets the property key handled by this permission.
	 *
	 * @return the property key.
	 */
	public String getKey() {
		return this.key;
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		return this == obj;
	}

	@Override
	public int hashCode() {
		return System.identityHashCode(this);
	}

	@Override
	public boolean implies(@Nullable Permission permission) {
		return equals(permission);
	}

}
