/*
 * Copyright 2018-2019 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.property;

import ej.property.loader.SystemPropertyLoader;
import ej.property.registry.SimplePropertyRegistry;
import ej.service.ServiceLoader;
import ej.service.loader.SystemPropertiesServiceLoader;

/**
 * Gets a property loader that is able to retrieve properties.
 * <p>
 * The property loader implementation is retrieved from system properties (see {@link SystemPropertiesServiceLoader}),
 * if none is found, a {@link SystemPropertyLoader} is used.
 * <p>
 * Usage:
 *
 * <pre>
 * String propertyValue = PropertyFactory.getPropertyLoader().getProperty("myKey");
 * </pre>
 *
 * <p>
 * The property registry implementation is retrieved from system properties (see {@link SystemPropertiesServiceLoader}),
 * if none is found, a {@link SimplePropertyRegistry} is used.
 * <p>
 * Usage:
 *
 * <pre>
 * PropertyFactory.getPropertyRegistry().put("myKey", "myValue");
 * </pre>
 */
public class PropertyFactory {

	// The property registry must be initialized before the property loader because the instance of the former can be
	// referenced by the instance of the latter.
	private static final PropertyRegistry PROPERTY_REGISTRY = getPropertyRegistryInstance();
	private static final PropertyLoader PROPERTY_LOADER = getPropertyLoaderInstance();

	private PropertyFactory() {
	}

	private static PropertyLoader getPropertyLoaderInstance() {
		ServiceLoader propertiesServiceLoader = new SystemPropertiesServiceLoader();
		PropertyLoader propertyLoaderCandidate = propertiesServiceLoader.getService(PropertyLoader.class);
		if (propertyLoaderCandidate != null) {
			return propertyLoaderCandidate;
		} else {
			return new SystemPropertyLoader();
		}
	}

	private static PropertyRegistry getPropertyRegistryInstance() {
		ServiceLoader propertiesServiceLoader = new SystemPropertiesServiceLoader();
		PropertyRegistry propertyRegistryCandidate = propertiesServiceLoader.getService(PropertyRegistry.class);
		if (propertyRegistryCandidate != null) {
			return propertyRegistryCandidate;
		} else {
			return new SimplePropertyRegistry();
		}
	}

	/**
	 * Gets a property loader unique instance.
	 *
	 * @return a property loader.
	 */
	public static PropertyLoader getPropertyLoader() {
		return PROPERTY_LOADER;
	}

	/**
	 * Gets a property registry unique instance.
	 *
	 * @return a property registry.
	 */
	public static PropertyRegistry getPropertyRegistry() {
		return PROPERTY_REGISTRY;
	}

}
