/*
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector;

import ej.annotation.Nullable;
import ej.mwt.Container;
import ej.mwt.Widget;

/**
 * A nth child selector selects by checking if a widget is the nth child of its parent.
 * <p>
 * Equivalent to <code>:nth-child</code> selector in CSS. Its specificity is (0,0,1,0).
 * <p>
 * The index is stored as a <code>byte</code> for heap optimization and therefore cannot exceed <code>255</code>.
 *
 * @see Widget#getParent()
 * @see SelectorHelper
 */
public class NthChildSelector implements Selector {

	private final byte index;

	/**
	 * Creates a nth child selector.
	 * <p>
	 * The given index is clamped between <code>0</code> and <code>255</code>.
	 *
	 * @param index
	 *            the index of the child.
	 */
	public NthChildSelector(int index) {
		assert (index >= 0 && index <= 0xFF) : "Index of nth child selector must be between 0 and 255.";
		this.index = (byte) index;
	}

	@Override
	public boolean appliesToWidget(Widget widget) {
		Container parent = widget.getParent();

		int index = (this.index & 0xFF);
		return parent != null && index < parent.getChildrenCount() && parent.getChild(index) == widget;
	}

	@Override
	public int getSpecificity() {
		return SelectorHelper.getSpecificity(0, 0, 1, 0);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && getClass() == obj.getClass()) {
			NthChildSelector other = (NthChildSelector) obj;
			return this.index == other.index;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return this.index;
	}
}
