/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui;

import ej.drawing.LLDWPainter;
import ej.microui.display.MicroUIGraphicsContext;
import ej.microui.display.MicroUIImage;
import ej.trace.Tracer;

/**
 * Utility class to log a MicroUI event: a drawing, a Buffer Refresh Strategy status, etc.
 * <p>
 * Log parameters are associated with the MicroUI SystemView description file <code>SYSVIEW_MicroUI.txt</code>. If an
 * event is malformed, the log will not be correctly decoded.
 */
public class Log {

	/**
	 * Event identifier for "flush".
	 */
	public static final int FLUSH = 16;

	/**
	 * Event identifier for "flush done".
	 */
	public static final int FLUSH_DONE = 17;

	/**
	 * Event identifier for "flush with a single rectangle".
	 */
	public static final int BRS_FLUSH_SINGLE_RECT = 51;

	/**
	 * Event identifier for "flush with several rectangles".
	 */
	public static final int BRS_FLUSH_MULTI_RECTS = 52;

	/**
	 * Event identifier for "restore Buffer Refresh Strategy region".
	 */
	public static final int BRS_RESTORE_REGION = 57;

	/**
	 * Event identifier for "string width".
	 */
	public static final int STRING_WIDTH = 62;

	/**
	 * Event identifier for "set foreground color".
	 */
	public static final int SET_FOREGROUND_COLOR = 63;

	/**
	 * Event identifier for "set background color".
	 */
	public static final int SET_BACKGROUND_COLOR = 64;

	/**
	 * Event identifier for "remove background color".
	 */
	public static final int REMOVE_BACKGROUND_COLOR = 65;

	/**
	 * Event identifier for "set clip".
	 */
	public static final int SET_CLIP = 66;

	/**
	 * Event identifier for "drawing out of clip".
	 */
	public static final int OUT_OF_CLIP = 67;

	/**
	 * Event identifier for "drawing size".
	 */
	public static final int DRAWN_REGION = 68;

	/**
	 * Event identifier for "draw a pixel".
	 */
	public static final int WRITE_PIXEL = 80;

	/**
	 * Event identifier for "draw a line".
	 */
	public static final int DRAW_LINE = 81;

	/**
	 * Event identifier for "draw an horizontal line".
	 */
	public static final int DRAW_HORIZONTAL_LINE = 82;

	/**
	 * Event identifier for "draw a vertical line".
	 */
	public static final int DRAW_VERTICAL_LINE = 83;

	/**
	 * Event identifier for "draw a rectangle".
	 */
	public static final int DRAW_RECTANGLE = 84;

	/**
	 * Event identifier for "fill a rectangle".
	 */
	public static final int FILL_RECTANGLE = 85;

	/**
	 * Event identifier for "draw a rounded rectangle".
	 */
	public static final int DRAW_ROUNDED_RECTANGLE = 86;

	/**
	 * Event identifier for "fill a rounded rectangle".
	 */
	public static final int FILL_ROUNDED_RECTANGLE = 87;

	/**
	 * Event identifier for "draw a circle arc".
	 */
	public static final int DRAW_CIRCLE_ARC = 88;

	/**
	 * Event identifier for "fill a circle arc".
	 */
	public static final int FILL_CIRCLE_ARC = 89;

	/**
	 * Event identifier for "draw an ellipse arc".
	 */
	public static final int DRAW_ELLIPSE_ARC = 90;

	/**
	 * Event identifier for "fill an ellipse arc".
	 */
	public static final int FILL_ELLIPSE_ARC = 91;

	/**
	 * Event identifier for "draw an ellipse".
	 */
	public static final int DRAW_ELLIPSE = 92;

	/**
	 * Event identifier for "fill an ellipse".
	 */
	public static final int FILL_ELLIPSE = 93;

	/**
	 * Event identifier for "draw a circle".
	 */
	public static final int DRAW_CIRCLE = 94;

	/**
	 * Event identifier for "fill a circle".
	 */
	public static final int FILL_CIRCLE = 95;

	/**
	 * Event identifier for "draw an image".
	 */
	public static final int DRAW_IMAGE = 96;

	/**
	 * Event identifier for "draw a string".
	 */
	public static final int DRAW_STRING = 97;

	/**
	 * Event identifier for "draw a renderable string".
	 */
	public static final int DRAW_RENDERABLE_STRING = 97;

	/**
	 * Event identifier for "draw a thick faded point".
	 */
	public static final int DRAW_THICK_FADED_POINT = 110;

	/**
	 * Event identifier for "draw a thick faded line".
	 */
	public static final int DRAW_THICK_FADED_LINE = 111;

	/**
	 * Event identifier for "draw a thick faded circle".
	 */
	public static final int DRAW_THICK_FADED_CIRCLE = 112;

	/**
	 * Event identifier for "draw a thick faded circle arc".
	 */
	public static final int DRAW_THICK_FADED_CIRCLE_ARC = 113;

	/**
	 * Event identifier for "draw a thick faded ellipse".
	 */
	public static final int DRAW_THICK_FADED_ELLIPSE = 114;

	/**
	 * Event identifier for "draw a thick line".
	 */
	public static final int DRAW_THICK_LINE = 115;

	/**
	 * Event identifier for "draw a thick circle".
	 */
	public static final int DRAW_THICK_CIRCLE = 116;

	/**
	 * Event identifier for "draw a thick ellipse".
	 */
	public static final int DRAW_THICK_ELLIPSE = 117;

	/**
	 * Event identifier for "draw a thick circle arc".
	 */
	public static final int DRAW_THICK_CIRCLEARC = 118;

	/**
	 * Event identifier for "draw a flipped image".
	 */
	public static final int DRAW_FLIPPED_IMAGE = 130;

	/**
	 * Event identifier for "draw a rotated image".
	 */
	public static final int DRAW_ROTATED_IMAGE = 131;

	/**
	 * Event identifier for "draw a scaled image".
	 */
	public static final int DRAW_SCALED_IMAGE = 132;

	/**
	 * Event identifier for "draw a scaled string".
	 */
	public static final int DRAW_SCALED_STRING = 133;

	/**
	 * Event identifier for "draw a scaled renderable string".
	 */
	public static final int DRAW_SCALED_RENDERABLE_STRING = 134;

	/**
	 * Event identifier for "draw a rotated character".
	 */
	public static final int DRAW_ROTATED_CHARACTER = 135;

	private static final int APPROXIMATION_BILINEAR = 0;
	private static final int APPROXIMATION_NEAREST_NEIGHBOR = 1;

	private static final Tracer MicroUITracer = new Tracer("MicroUI", 200); //$NON-NLS-1$

	private Log() {
		// cannot instantiate
	}

	/**
	 * Records a MicroUI event. The given event identifier must be valid for this logger (i.e. between
	 * {@link #BRS_FLUSH_SINGLE_RECT} and {@link #DRAW_ROTATED_CHARACTER}).
	 *
	 * @param eventId
	 *            the event identifier
	 * @param values
	 *            the event parameters
	 */
	public static void recordEvent(int eventId, int... values) {
		MicroUITracer.recordEvent(eventId, values);
	}

	/**
	 * Records the end of the execution of a MicroUI event. Call this method to trace the duration of an event
	 * previously record with one of the {@link #recordEventEnd(int, int...)} method. This method should not be called
	 * for instantaneous events.
	 *
	 * @param eventId
	 *            the "end" event identifier
	 * @param values
	 *            the "end" event parameters
	 */
	public static void recordEventEnd(int eventId, int... values) {
		MicroUITracer.recordEventEnd(eventId, values);
	}

	/**
	 * Maps a graphics context to an event parameter.
	 *
	 * @param gc
	 *            the graphics context to log
	 * @return the unique graphics context identifier
	 */
	public static int mapGraphicsContext(MicroUIGraphicsContext gc) {
		return gc.getIdentifier();
	}

	/**
	 * Maps an image to an event parameter.
	 *
	 * @param img
	 *            the image to log
	 * @return the unique image identifier
	 */
	public static int mapImage(MicroUIImage img) {
		return img.getIdentifier();
	}

	/**
	 * Maps a boolean to an event parameter.
	 *
	 * @param value
	 *            the boolean to log
	 * @return {@code 1} for {@code true}, {@code 0} for {@code false}
	 */
	public static int mapBoolean(boolean value) {
		return value ? 1 : 0;
	}

	/**
	 * Maps the approximation bilinear or nearest neighbor used by some {@link LLDWPainter}
	 * algorithms
	 *
	 * @param bilinear
	 *            {@code true} for bilinear or {@code false} for nearest neighbor
	 * @return {@code 1} for nearest neighbor, {@code 0} for bilinear
	 * @see #DRAW_ROTATED_IMAGE
	 * @see #DRAW_SCALED_IMAGE
	 * @see #DRAW_ROTATED_CHARACTER
	 * @see #DRAW_SCALED_STRING
	 * @see #DRAW_SCALED_RENDERABLE_STRING
	 */
	public static int mapApproximation(boolean bilinear) {
		return bilinear ? APPROXIMATION_BILINEAR : APPROXIMATION_NEAREST_NEIGHBOR;
	}
}
