/*
 * Copyright 2019-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */

package ej.net.util.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Properties;
import java.util.Set;

/**
 * Manages the certificates of the device.
 */
public class CertificatesHelper {

	private static final String CERTIFICATE_TYPE = "X.509";

	private CertificatesHelper() {
		// private constructor
	}

	/**
	 * Loads the X.509 certificate from the given input stream.
	 *
	 * @param inputStream
	 *            the input stream.
	 * @return the certificate.
	 * @throws CertificateException
	 *             if the input stream does not contain a valid certificate.
	 */
	public static X509Certificate loadCertificate(InputStream inputStream) throws CertificateException {
		CertificateFactory certificateFactory;
		try {
			certificateFactory = CertificateFactory.getInstance(CERTIFICATE_TYPE);
		} catch (CertificateException e) {
			throw new UnsupportedOperationException(e);
		}

		return (X509Certificate) certificateFactory.generateCertificate(inputStream);
	}

	/**
	 * Gets the X.509 certificate from the resource at the given path.
	 *
	 * @param path
	 *            the resource path.
	 * @return the certificate.
	 * @throws IllegalArgumentException
	 *             if the resource does not exist or it does not contain a valid certificate.
	 */
	public static X509Certificate getCertificate(String path) {
		try (InputStream inputStream = CertificatesHelper.class.getResourceAsStream(path)) {
			if (inputStream == null) {
				throw new IllegalArgumentException("Cannot get resource \"" + path + "\"");
			}
			return loadCertificate(inputStream);
		} catch (CertificateException | IOException e) {
			throw new IllegalArgumentException("Cannot generate certificate from resource \"" + path + "\"", e);
		}
	}

	/**
	 * Loads a list of certificates by reading an input stream which contains the filename of each certificate.
	 *
	 * @param certificatesList
	 *            the input stream containing the list of certificates.
	 * @return an array containing the filenames of the certificates.
	 * @throws IOException
	 *             if an exception occurs while loading the certificates list.
	 */
	public static String[] loadCertificatesList(InputStream certificatesList) throws IOException {
		Properties properties = new Properties();
		properties.load(certificatesList);
		Set<String> propertiesSet = properties.stringPropertyNames();
		return propertiesSet.toArray(new String[propertiesSet.size()]);
	}
}
