/*
 * Copyright 2013-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.widget.util.color;

import ej.bon.XMath;

/**
 * Color utilities.
 */
public class ColorHelper {

	private static final int MAX_COLOR_COMPONENT_VALUE = 0xff;
	private static final int RED_SHIFT = 16;
	private static final int GREEN_SHIFT = 8;
	private static final int BLUE_SHIFT = 0;
	private static final int RED_MASK = MAX_COLOR_COMPONENT_VALUE << RED_SHIFT;
	private static final int GREEN_MASK = MAX_COLOR_COMPONENT_VALUE << GREEN_SHIFT;
	private static final int BLUE_MASK = MAX_COLOR_COMPONENT_VALUE << BLUE_SHIFT;

	// Prevents initialization.
	private ColorHelper() {
	}

	/**
	 * Gets a color red component.
	 *
	 * @param color
	 *            the color.
	 * @return the red component.
	 */
	public static int getRed(int color) {
		return (color & RED_MASK) >>> RED_SHIFT;
	}

	/**
	 * Gets a color green component.
	 *
	 * @param color
	 *            the color.
	 * @return the green component.
	 */
	public static int getGreen(int color) {
		return (color & GREEN_MASK) >>> GREEN_SHIFT;
	}

	/**
	 * Gets a color blue component.
	 *
	 * @param color
	 *            the color.
	 * @return the blue component.
	 */
	public static int getBlue(int color) {
		return (color & BLUE_MASK) >>> BLUE_SHIFT;
	}

	/**
	 * Gets the color from red, green and blue components.
	 *
	 * @param red
	 *            the red component.
	 * @param green
	 *            the green component.
	 * @param blue
	 *            the blue component.
	 * @return the color.
	 */
	public static int getColor(int red, int green, int blue) {
		return (red << RED_SHIFT) & RED_MASK | (green << GREEN_SHIFT) & GREEN_MASK | (blue << BLUE_SHIFT) & BLUE_MASK;
	}

	/**
	 * Update a color component by incrementing it.
	 * <p>
	 * The resulting value is bounded to a component limits.
	 *
	 * @param component
	 *            the initial component value.
	 * @param increment
	 *            the increment to apply.
	 * @return the incremented component.
	 */
	public static int updateComponent(int component, int increment) {
		return XMath.limit(component + increment, 0x0, MAX_COLOR_COMPONENT_VALUE);
	}

}