/*
 * Java
 *
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http;

import static ej.hoka.http.HttpConstants.ALLOW;
import static ej.hoka.http.HttpConstants.CLOSE;
import static ej.hoka.http.HttpConstants.EMPTY;
import static ej.hoka.http.HttpConstants.HEADER_ACCEPT_ENCODING;
import static ej.hoka.http.HttpConstants.HEADER_CONNECTION;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_BADREQUEST;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_INTERNALERROR;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_METHODNOTALLOWED;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_NOTACCEPTABLE;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_NOTFOUND;
import static ej.hoka.http.HttpConstants.HTTP_STATUS_REQUESTTIMEOUT;
import static ej.hoka.http.HttpConstants.TAB;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.Date;

import ej.annotation.Nullable;
import ej.hoka.http.encoding.ContentEncoding;
import ej.hoka.http.encoding.EncodingRegistry;
import ej.hoka.http.encoding.HttpUnsupportedEncodingException;
import ej.hoka.http.support.Mime;
import ej.hoka.log.HokaLogger;
import ej.hoka.tcp.TcpServer;

/**
 * HTTP worker to process incoming connections. This worker needs to be thread safe.
 */
class Worker implements Runnable {

	/**
	 * Unknown/invalid IP address.
	 */
	private static final String UNKNOWN_ADDRESS = "0.0.0.0"; //$NON-NLS-1$

	/**
	 * The HTML line break tag.
	 */
	private static final String HTML_BR = "<br/>"; //$NON-NLS-1$

	/**
	 * The HTML paragraph start tag.
	 */
	private static final String HTML_PARAGRAPH_START = "<p>"; //$NON-NLS-1$

	/**
	 * The HTML paragraph closing tag.
	 */
	private static final String HTML_PARAGRAPH_CLOSE = "</p>"; //$NON-NLS-1$

	private final TcpServer server;
	private final RouteHandler routesHandler;
	/**
	 * Activate development mode to send Java exception stack trace to the client.
	 */
	private final boolean devMode;

	/**
	 * Activate strict encoding support. RFC2616 14.3
	 */
	private final boolean strictAcceptEncoding;

	private final EncodingRegistry encodingRegistry;

	/**
	 * not found exception message and mime type
	 */
	@Nullable
	private final String notFoundErrorMessage;
	private final String notFoundErrorContentType;

	/**
	 * internal server exception message and mime type
	 */
	private final String internalServerErrorMessage;
	private final String internalServerErrorContentType;

	/**
	 * Construct an HTTP Worker
	 *
	 * @param server
	 * @param routesHandler
	 * @param encodingRegistry
	 * @param strictAcceptEncoding
	 * @param notFoundErrorMessage
	 * @param notFoundErrorContentType
	 * @param internalServerErrorMessage
	 * @param internalServerErrorContentType
	 * @param devMode
	 */
	Worker(final TcpServer server, final RouteHandler routesHandler, final EncodingRegistry encodingRegistry, // NOSONAR
			boolean strictAcceptEncoding, final String notFoundErrorMessage, final String notFoundErrorContentType,
			final String internalServerErrorMessage, String internalServerErrorContentType, boolean devMode) {
		this.server = server;
		this.routesHandler = routesHandler;
		this.encodingRegistry = encodingRegistry;
		this.strictAcceptEncoding = strictAcceptEncoding;
		this.notFoundErrorMessage = notFoundErrorMessage;
		this.notFoundErrorContentType = notFoundErrorContentType;
		this.internalServerErrorMessage = internalServerErrorMessage;
		this.internalServerErrorContentType = internalServerErrorContentType;
		this.devMode = devMode;
	}

	@Override
	public void run() {
		while (true) {
			try (Socket connection = this.server.getNextStreamConnection()) {

				if (connection == null) {
					return;// server stopped
				}

				HokaLogger.instance.trace(connection.hashCode() + TAB + getHostAdressString(connection.getInetAddress())
						+ "\t processing"); //$NON-NLS-1$

				processConnection(connection);

				HokaLogger.instance.trace(
						connection.hashCode() + TAB + getHostAdressString(connection.getInetAddress()) + "\t closed"); //$NON-NLS-1$

			} catch (IOException e) {
				HokaLogger.instance.error(e);
			}
		}
	}

	/**
	 * Process request.
	 *
	 * @param connection
	 *            socket
	 */
	private void processConnection(final Socket connection) {

		final InetAddress address = connection.getInetAddress();
		final int hash = connection.hashCode();

		try (InputStream inputStream = new BufferedInputStream(connection.getInputStream(),
				Config.getInstance().getBufferSize()); OutputStream outputStream = connection.getOutputStream()) {

			doProcess(inputStream, outputStream, address, hash);

		} catch (final IOException e) {
			// connection lost

			HokaLogger.instance
					.error(connection.hashCode() + TAB + (address != null ? address.toString() : UNKNOWN_ADDRESS), e);
		}
	}

	/**
	 * handle request / response per connection.
	 *
	 * This method parse the {@link HttpRequest} from the input stream and create the {@link HttpResponse} instance
	 *
	 * then, it delegate the request/response processing to the router.
	 *
	 * at the end the response is written into the request output stream
	 *
	 *
	 * @param inputStream
	 *            request input stream
	 * @param outputStream
	 *            request output stream
	 * @param address
	 *            connection address
	 * @param hash
	 *            socket hash code for log
	 * @throws IOException
	 *             on error
	 */
	private void doProcess(final InputStream inputStream, final OutputStream outputStream,
			@Nullable final InetAddress address, final int hash) throws IOException {

		HttpRequest request = null;
		final HttpResponse response = new HttpResponse();
		ContentEncoding encodingHandler = null;

		try {

			request = new HttpRequest(inputStream, this.encodingRegistry);

			// RFC2616 14.3
			if (this.strictAcceptEncoding) {
				String accept = request.getHeader(HEADER_ACCEPT_ENCODING);
				encodingHandler = this.encodingRegistry.getAcceptEncodingHandler(accept);
				if (encodingHandler == null) {
					throw new HttpUnsupportedEncodingException(HTTP_STATUS_NOTACCEPTABLE, accept);
				}
			}

			this.routesHandler.process(request, response);

		} catch (final HaltException e) { // NOSONAR
			response.setStatus(e.getStatus());
			response.setData(e.getBody());
		} catch (final MethodNotAllowedException e) {
			handleMethodNotAllowedError(response, e); // 405
		} catch (final RouteNotFoundException e) { // NOSONAR
			handleNotFoundError(response); // 404
		} catch (final IllegalArgumentException e) {
			handleError(response, HTTP_STATUS_BADREQUEST, e);
		} catch (final HttpUnsupportedEncodingException e) {
			handleError(response, HTTP_STATUS_NOTACCEPTABLE, e);
		} catch (final SocketTimeoutException e) {
			handleError(response, HTTP_STATUS_REQUESTTIMEOUT, e);
		} catch (final IOException e) {
			throw e;
		} catch (final Exception e) {
			handleInternalServerError(response, e);
		}

		// keep-alive is not supported. inform the client that the connection is closed
		response.addHeader(HEADER_CONNECTION, CLOSE);

		// log the response
		final String msg = createResponseLogMsg(request == null ? EMPTY : request.getURI(), response.getStatus(), hash,
				getHostAdressString(address));
		HokaLogger.instance.debug(msg);

		// send response back
		response.sendResponse(outputStream, encodingHandler, this.encodingRegistry,
				Config.getInstance().getBufferSize());
	}

	private static String getHostAdressString(@Nullable final InetAddress address) {
		return address != null ? address.getHostAddress() : UNKNOWN_ADDRESS;
	}

	private String createResponseLogMsg(String uri, String status, int connection, String address) {
		StringBuilder msg = new StringBuilder();
		msg.append(connection);
		msg.append(TAB);
		msg.append(address);
		msg.append(TAB);
		msg.append(new Date());
		msg.append(TAB);
		msg.append(uri);
		msg.append(TAB);
		msg.append(status);
		return msg.toString();
	}

	/**
	 * Set response to method not allowed with allow header
	 *
	 * @param response
	 *            http response
	 * @param e
	 *            MethodNotAllowedException
	 */
	private void handleMethodNotAllowedError(HttpResponse response, MethodNotAllowedException e) {
		response.setStatus(HTTP_STATUS_METHODNOTALLOWED);
		response.setData(EMPTY);
		response.addHeader(ALLOW, e.getAllowHeader());
	}

	/**
	 * @param response
	 * @param e
	 */
	private void handleInternalServerError(final HttpResponse response, final Throwable e) {
		response.setStatus(HTTP_STATUS_INTERNALERROR);
		if (this.devMode) {
			response.setMimeType(Mime.MIME_HTML);
			response.setData(createHtmlError(HTTP_STATUS_INTERNALERROR, e.getMessage(), e));
		} else {
			response.setMimeType(this.internalServerErrorContentType);
			response.setData(this.internalServerErrorMessage);
		}
	}

	private void handleError(final HttpResponse response, final String status, final Exception e) {
		response.setStatus(status);
		if (this.devMode) {
			response.setMimeType(Mime.MIME_HTML);
			response.setData(createHtmlError(status, e.getMessage(), e));
		} else {
			response.setData(EMPTY);
		}
	}

	private void handleNotFoundError(final HttpResponse response) {
		response.setStatus(HTTP_STATUS_NOTFOUND);
		response.setMimeType(this.notFoundErrorContentType);
		final String errorMessage = this.notFoundErrorMessage;
		if (errorMessage == null || errorMessage.isEmpty()) {
			response.setData(EMPTY);
		} else {
			response.setData(errorMessage);
		}
	}

	/**
	 * Create a {@link HttpResponse} to write the <code>msg</code> for the given <code>status</code>.
	 *
	 * @param status
	 *            the error status. One of <code>HTTP_STATUS_*</code> constant of the {@link HttpConstants} interface.
	 * @param msg
	 *            an optional error message to add in response.
	 * @return a {@link HttpResponse} that represent the error.
	 * @see HttpConstants#HTTP_STATUS_BADREQUEST
	 * @see HttpConstants#HTTP_STATUS_FORBIDDEN
	 * @see HttpConstants#HTTP_STATUS_INTERNALERROR
	 * @see HttpConstants#HTTP_STATUS_MEDIA_TYPE
	 * @see HttpConstants#HTTP_STATUS_METHODNOTALLOWED
	 * @see HttpConstants#HTTP_STATUS_NOTACCEPTABLE
	 * @see HttpConstants#HTTP_STATUS_NOTFOUND
	 * @see HttpConstants#HTTP_STATUS_NOTIMPLEMENTED
	 * @see HttpConstants#HTTP_STATUS_NOTMODIFIED
	 * @see HttpConstants#HTTP_STATUS_OK
	 * @see HttpConstants#HTTP_STATUS_REDIRECT
	 */
	private static String createHtmlError(String status, @Nullable String msg, @Nullable Throwable t) {
		StringBuilder buffer = new StringBuilder();
		buffer.append("<html><head><title>"); //$NON-NLS-1$
		buffer.append(status);
		buffer.append("</title></head><body>"); //$NON-NLS-1$
		buffer.append("<h3>").append(status).append("</h3>"); //$NON-NLS-1$ //$NON-NLS-2$
		if (msg != null && !msg.isEmpty()) {
			buffer.append(HTML_PARAGRAPH_START).append(msg).append(HTML_PARAGRAPH_CLOSE);
		}
		if (t != null) {
			buffer.append("<hr/>"); //$NON-NLS-1$
			buffer.append("<b>Exception:</b>"); //$NON-NLS-1$
			buffer.append(HTML_PARAGRAPH_START).append(getHtmlExceptionStackTrace(t)).append(HTML_PARAGRAPH_CLOSE);
		}
		buffer.append("</body></html>"); //$NON-NLS-1$
		return buffer.toString();
	}

	/**
	 * Creates a HTML representation of the stack trace of <code>t</code>.
	 * <p>
	 * Only the relevant part of the stack trace is dumped. The exception superclasses constructors and the job internal
	 * calls are skipped.
	 *
	 * @param t
	 *            the throwable to dump.
	 * @return the HTML representation of the stack trace as a {@link String}.
	 */
	private static String getHtmlExceptionStackTrace(Throwable t) {
		StringBuilder fullMessageBuilder = new StringBuilder();

		String message = t.getMessage();
		if (message != null) {
			fullMessageBuilder.append(message);
			fullMessageBuilder.append(HTML_BR);
		}

		StackTraceElement[] stackTrace = t.getStackTrace();

		int i = 0;
		String className;

		// Skip all the exception superclasses constructors
		className = "java.lang"; //$NON-NLS-1$
		while (i < stackTrace.length && stackTrace[i].getClassName().startsWith(className)) {
			i++;
		}

		// Append only the stack trace up to this class call to RequestHandler#process.
		if (i > 0 && i - 1 < stackTrace.length) {
			fullMessageBuilder.append(stackTrace[i - 1].toString()).append(HTML_BR);
		}

		className = HttpServer.class.getName();
		final String prefix = "&nbsp;&nbsp;&nbsp;&nbsp;at "; //$NON-NLS-1$
		for (; i < stackTrace.length && !stackTrace[i].getClassName().equals(className); i++) {
			fullMessageBuilder.append(prefix).append(stackTrace[i].toString()).append(HTML_BR);
		}

		return fullMessageBuilder.toString();
	}
}
