/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Tests kerning with transform and gradient.
 */
@SuppressWarnings("nls")
public class TestFontComplexLayoutGradient {

	private static final String AMIRI_FONT = "/fonts/Amiri-reduced.ttf";

	private static final String ARABIC_STRING = "مظللمظللمظلل";

	/**
	 * Offset for gradient xStart and xEnd (in pixels).
	 */
	private static final int GRADIENT_START_END = 5;

	/**
	 * @see #check(int[][], int, int, int, int, int)
	 */
	private static final int DEFAULT_TOLERANCE = 5;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test drawing an arabic string with a linear gradient
	 */
	@Test
	public void testDrawGradientString() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT, true);

		int fontSize = 30;
		float stringWidth = font.measureStringWidth(ARABIC_STRING, fontSize);
		float stringHeight = font.measureStringHeight(ARABIC_STRING, fontSize);

		LinearGradient gradient = new LinearGradient(GRADIENT_START_END, 0, stringWidth - GRADIENT_START_END, 0,
				new int[] { 0xffff0000, 0xff00ff00, 0xff0000ff });

		Matrix m = new Matrix();
		VectorGraphicsPainter.drawGradientString(g, ARABIC_STRING, font, fontSize, m, gradient, 0xFF,
				BlendMode.SRC_OVER, 0);
		display.flush();

		int w = (int) (stringWidth - 2);
		int[][] colors = getComponents(g, 1, 16, w, (int) stringHeight);

		int gradientSize = w - 2 * GRADIENT_START_END;

		// "red" decreases between 0 to w / 2 and is stable between w/2 and w
		check(colors, 0, 0, w / 2, -gradientSize / 2);
		check(colors, 0, w / 2, w / 2, 0);

		// "green" increases between 0 to w / 2 and decreases between w/2 and w
		check(colors, 1, 0, w / 2, gradientSize / 2);
		check(colors, 1, w / 2, w / 2, -gradientSize / 2);

		// "blue" is stable between 0 to w / 2 and increases between w/2 and w
		check(colors, 2, 0, w / 2, 0);
		check(colors, 2, w / 2, w / 2, gradientSize / 2);
	}

	/**
	 * Test drawing an arabic string with a linear gradient and a rotation
	 */
	@Test
	public void testDrawGradientStringAndRotation() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT, true);

		int fontSize = 30;
		float stringWidth = font.measureStringWidth(ARABIC_STRING, fontSize);
		float stringHeight = font.measureStringHeight(ARABIC_STRING, fontSize);

		LinearGradient gradient = new LinearGradient(GRADIENT_START_END, 0, stringWidth - GRADIENT_START_END, 0,
				new int[] { 0xffff0000, 0xff00ff00, 0xff0000ff });

		Matrix m = new Matrix();
		m.preTranslate(200, 100);
		m.preRotate(180);
		VectorGraphicsPainter.drawGradientString(g, ARABIC_STRING, font, fontSize, m, gradient, 0xFF,
				BlendMode.SRC_OVER, 0);
		display.flush();

		int w = (int) (stringWidth - 2);
		int[][] colors = getComponents(g, 79, 51, w, (int) stringHeight);

		int gradientSize = w - 2 * GRADIENT_START_END;

		// "blue" decreases between 0 to w / 2 and is stable between w/2 and w
		check(colors, 2, 0, w / 2, -gradientSize / 2);
		check(colors, 2, w / 2, w / 2, 0);

		// "green" increases between 0 to w / 2 and decreases between w/2 and w
		check(colors, 1, 0, w / 2, gradientSize / 2);
		check(colors, 1, w / 2, w / 2, -gradientSize / 2);

		// "red" is stable between 0 to w / 2 and increases between w/2 and w
		check(colors, 0, 0, w / 2, 0);
		check(colors, 0, w / 2, w / 2, gradientSize / 2);

	}

	/**
	 * Test drawing an arabic string with a linear gradient and a scaling
	 */
	@Test
	public void testDrawGradientStringAndScale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT, true);

		int fontSize = 30;
		float stringWidth = font.measureStringWidth(ARABIC_STRING, fontSize);
		float stringHeight = font.measureStringHeight(ARABIC_STRING, fontSize);

		LinearGradient gradient = new LinearGradient(GRADIENT_START_END, 0, stringWidth - GRADIENT_START_END, 0,
				new int[] { 0xffff0000, 0xff00ff00, 0xff0000ff });

		Matrix m = new Matrix();
		m.preScale(1.5f, 1);
		VectorGraphicsPainter.drawGradientString(g, ARABIC_STRING, font, fontSize, m, gradient, 0xFF,
				BlendMode.SRC_OVER, 0);
		display.flush();

		int w = (int) (stringWidth * 1.5f - 2);
		int[][] colors = getComponents(g, 2, 17, w, (int) stringHeight);

		int gradientSize = w - 2 * GRADIENT_START_END;

		// "red" decreases between 0 to w / 2 and is stable between w/2 and w
		check(colors, 0, 0, w / 2, -gradientSize / 2);
		check(colors, 0, w / 2, w / 2, 0);

		// "green" increases between 0 to w / 2 and decreases between w/2 and w
		check(colors, 1, 0, w / 2, gradientSize / 2);
		check(colors, 1, w / 2, w / 2, -gradientSize / 2);

		// "blue" is stable between 0 to w / 2 and increases between w/2 and w
		check(colors, 2, 0, w / 2, 0);
		check(colors, 2, w / 2, w / 2, gradientSize / 2);
	}

	/**
	 * Test drawing an arabic string with a linear gradient on a circle
	 */
	@Test
	public void testDrawGradientStringOnCircle() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT, true);

		String string = ARABIC_STRING + ARABIC_STRING + ARABIC_STRING;
		final int radius = 50;
		final int fontSize = 30;

		// Colors must be provided with alpha channel
		LinearGradient gradient = new LinearGradient(0, -radius + GRADIENT_START_END, 0, radius - GRADIENT_START_END,
				new int[] { 0xffff0000, 0xff00ff00, 0xff0000ff });

		Matrix matrix = new Matrix();
		matrix.setRotate(-90);
		matrix.postTranslate(radius, radius);

		// draw the same string with both directions to accentuate the colors
		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.COUNTER_CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		int w = radius * 2;
		int[][] colors = getComponents(g, 3, 3, w, w);

		int gradientSize = w - 2 * GRADIENT_START_END;

		display.flush();

		// "red" decreases between 0 to radius and is stable between radius and w
		check(colors, 0, 0, radius, -gradientSize / 2, 8);
		check(colors, 0, radius, radius, 0, 8);

		// "green" increases between 0 to radius and decreases between radius and w
		check(colors, 1, 0, radius, gradientSize / 2, 8);
		check(colors, 1, radius, radius, -gradientSize / 2, 8);

		// "blue" is stable between 0 to radius and increases between radius and w
		check(colors, 2, 0, radius, 0, 8);
		check(colors, 2, radius, radius, gradientSize / 2, 8);
	}

	/**
	 * Gets the "highest" value of each component (red, green, blue) on each vertical line.
	 *
	 * @return an array of array: [0] for red, [1] for green and [2] for blue; the sub array is the "highest" value for
	 *         each column
	 */
	private int[][] getComponents(GraphicsContext gc, int x, int y, int w, int h) {

		// read the pixels
		int[] pixels = new int[w * h];
		gc.readPixels(pixels, 0, w, x, y, w, h);

		// keep the "highest" value of each component (red, green, blue) on each v line
		int[][] colors = new int[3][w]; // 0 init
		for (x = 0; x < w; x++) {

			int red = 0;
			int blue = 0;
			int green = 0;

			for (y = 0; y < h; y++) {
				int c = pixels[x + y * w];
				red = Math.max(red, (c >> 16) & 0xff);
				green = Math.max(green, (c >> 8) & 0xff);
				blue = Math.max(blue, (c >> 0) & 0xff);
			}

			colors[0][x] = red;
			colors[1][x] = green;
			colors[2][x] = blue;
		}

		return colors;
	}

	/**
	 * Checks if the horizontal line increases or decreases or stays unchanged.
	 *
	 * @param colors
	 *            array returned by {@link #getComponents(GraphicsContext, int, int, int, int)}
	 * @param col
	 *            0 for red, 1 for green, 2 for blue
	 * @param off
	 *            the first x position
	 * @param size
	 *            the width to check
	 * @param way
	 *            the size (in number of pixels) of the incrementation (positive) or of the decrementation (negative) or
	 *            of the stability.
	 * @param tolerance
	 *            value to determinate if a color component increase or decrease; according to the drawing and the
	 *            anti-aliasing, a color component can decrease, then increase for a short time and decrease again: the
	 *            main way is "decrease".
	 */
	private void check(int[][] colors, int col, int off, int size, int way, int tolerance) {
		int bigger = 0;
		int smaller = 0;
		int same = 0;
		for (int x = off + 1; x < off + size; x++) {
			int color = colors[col][x];
			int previousColor = colors[col][x - 1];
			if (color > previousColor) {
				++bigger;
			} else if (color < previousColor) {
				++smaller;
			} else {
				++same;
			}
		}

		System.out.println("colour " + col + " [" + off + ", " + (off + size - 1) + "]");
		System.out.println("\tbigger\tsmaller\tsame");
		System.out.print("\t" + bigger);
		System.out.print("\t" + smaller);
		System.out.println("\t" + same);

		if (way < 0) {
			// decrease expected
			Assert.assertTrue("decrease expected on column " + col,
					(smaller > bigger) && (bigger <= tolerance) && ((smaller + same) > (-way - tolerance)));
		} else if (way == 0) {
			// stable expected
			Assert.assertTrue("stable expected on column " + col,
					(Math.abs(smaller - bigger) <= tolerance) && (smaller <= tolerance) && (bigger <= tolerance));
		} else {
			// increase expected
			Assert.assertTrue("increase expected on column " + col,
					(smaller < bigger) && (smaller <= tolerance) && ((bigger + same) > (way - tolerance)));
		}
	}

	private void check(int[][] colors, int col, int off, int size, int way) {
		check(colors, col, off, size, way, DEFAULT_TOLERANCE);
	}
}
