/*
 * Copyright 2022-2023 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.android.application;

import android.content.Context;

/**
 * This interface can be used by Android Applications in order to setup the MicroEJ support engine and to handle display
 * events.
 */
public interface ApplicationSupport {

	/**
	 * Starts up the MicroEJ support engine for Android.
	 */
	public void startup();

	/**
	 * Sets the Android application context that should be used by the MicroEJ support engine.
	 *
	 * @param context
	 *            the context of the Android Application.
	 */
	public void setContext(Context context);

	/**
	 * Sets the current application display.
	 *
	 * @param display
	 *            the display to set.
	 */
	public void setDisplay(ApplicationDisplay display);

	/**
	 * Unsets the current application display.
	 * <p>
	 * This method does nothing if the current application display is not the given object.
	 *
	 * @param display
	 *            the display to unset.
	 */
	public void unsetDisplay(ApplicationDisplay display);

	/**
	 * Handles the tearing signal of the current application display.
	 * <p>
	 * This method does nothing if the current application display is not the given object.
	 *
	 * @param display
	 *            the display performing the tearing signal.
	 */
	void onDisplayTearing(ApplicationDisplay display);

	/**
	 * Handles a pointer press of the current application display.
	 * <p>
	 * This method does nothing if the current application display is not the given object.
	 *
	 * @param display
	 *            the display notifying the pointer press.
	 * @param x
	 *            the x coordinate of the pointer.
	 * @param y
	 *            the y coordinate of the pointer.
	 */
	void onPointerPress(ApplicationDisplay display, int x, int y);

	/**
	 * Handles a pointer release of the current application display.
	 * <p>
	 * This method does nothing if the current application display is not the given object.
	 *
	 * @param display
	 *            the display notifying the pointer release.
	 */
	void onPointerRelease(ApplicationDisplay display);

	/**
	 * Handles a pointer move of the current application display.
	 * <p>
	 * This method does nothing if the current application display is not the given object.
	 *
	 * @param display
	 *            the display notifying the pointer move.
	 * @param x
	 *            the x coordinate of the pointer.
	 * @param y
	 *            the y coordinate of the pointer.
	 */
	void onPointerMove(ApplicationDisplay display, int x, int y);
}
