/*
 * Copyright 2013-2014 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util.logging;

/**
 * LogRecord objects are used to pass logging requests between the logging framework and individual log Handlers.
 * <p>
 * When a LogRecord is passed into the logging framework it logically belongs to the framework and should no longer be used or updated by
 * the client application.
 * <p>
 * Note that if the client application has not specified an explicit source method name and source class name, then the LogRecord class will
 * infer them automatically when they are first accessed (due to a call on getSourceMethodName or getSourceClassName) by analyzing the call
 * stack. Therefore, if a logging Handler wants to pass off a LogRecord to another thread, or to transmit it over RMI, and if it wishes to
 * subsequently obtain method name or class name information it should call one of getSourceClassName or getSourceMethodName to force the
 * values to be filled in.
 */
public class LogRecord {

	// Current sequence number that will be assigned to the next LogRecord
	private static long CurrentSequenceNumber = 0;

	// Properties of this log record
	private final long sequenceNumber;
	private long millis;

	private Level level;
	private String message;

	private Throwable thrown;
	private String loggerName;

	/**
	 * Construct a LogRecord with the given level and message.
	 * 
	 * @param level
	 *            a logging level value.
	 * @param msg
	 *            the logging message.
	 */
	public LogRecord(Level level, String msg) {
		this.level = level;
		this.message = msg;

		// Generated fields
		this.sequenceNumber = CurrentSequenceNumber++;
		// FIXME thread safety issue
		this.millis = System.currentTimeMillis();

		// Initialize other properties to "null"
		thrown = null;
		loggerName = null;
	}

	/**
	 * Get the name of the logger that logged this LogRecord.
	 * 
	 * @return source logger name (may be {@code null}).
	 */
	public String getLoggerName() {
		return loggerName;
	}

	/**
	 * Set the source logger's name.
	 * 
	 * @param name
	 *            the source logger's name.
	 */
	public void setLoggerName(String name) {
		this.loggerName = name;
	}

	/**
	 * Get the logging message level.
	 * 
	 * @return the logging message level.
	 */
	public Level getLevel() {
		return level;
	}

	/**
	 * Set the logging message level.
	 * 
	 * @param level
	 *            the logging message level.
	 */
	public void setLevel(Level level) {
		this.level = level;
	}

	/**
	 * Get the sequence number of the LogRecord. This number is normally a unique identifier assigned by the constructor.
	 * 
	 * @return the sequence number of the LogRecord.
	 */
	public long getSequenceNumber() {
		return sequenceNumber;
	}

	/**
	 * Get the raw log message.
	 * 
	 * @return the message string.
	 */
	public String getMessage() {
		return message;
	}

	/**
	 * Set the raw log message.
	 * 
	 * @param message
	 *            the message to set.
	 */
	public void setMessage(String message) {
		this.message = message;
	}

	/**
	 * Get event time in milliseconds since 1970. This is the creation time of the LogRecord.
	 * 
	 * @return event time in milliseconds.
	 */
	public long getMillis() {
		return millis;
	}

	/**
	 * Set event time of the LogRecord.
	 * 
	 * @param millis
	 *            event time.
	 */
	public void setMillis(long millis) {
		this.millis = millis;
	}

	/**
	 * Get the throwable associated with this log record, if any.
	 * 
	 * @return a throwable
	 */
	public Throwable getThrown() {
		return thrown;
	}

	/**
	 * Set the throwable associated with this log record.
	 * 
	 * @param thrown
	 *            a throwable
	 */
	public void setThrown(Throwable thrown) {
		this.thrown = thrown;
	}

}
