/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth.listeners;

import ej.bluetooth.BluetoothAddress;
import ej.bluetooth.BluetoothConnection;
import ej.bluetooth.BluetoothService;

/**
 * The <code>ConnectionListener</code> interface provides callbacks related to advertising, scanning and connections.
 * <p>
 * Before {@link ej.bluetooth.BluetoothAdapter#startAdvertising(byte[]) starting to advertise} or before
 * {@link ej.bluetooth.BluetoothAdapter#startScanning(ej.bluetooth.BluetoothScanFilter) starting to scan}, the
 * application may {@link ej.bluetooth.BluetoothAdapter#setConnectionListener set the connection listener} of the
 * adapter in order to be notified of the events related to this adapter.
 * <p>
 * This library provides a default implementation of this interface:
 * {@link ej.bluetooth.listeners.impl.DefaultConnectionListener}.
 */
public interface ConnectionListener {

	/**
	 * Called when an ongoing advertisement is completed.
	 *
	 * @see ej.bluetooth.BluetoothAdapter#startAdvertising(byte[])
	 * @see ej.bluetooth.BluetoothAdapter#stopAdvertising()
	 */
	void onAdvertisementCompleted();

	/**
	 * Called when an ongoing scan is completed.
	 *
	 * @see ej.bluetooth.BluetoothAdapter#startScanning(ej.bluetooth.BluetoothScanFilter)
	 * @see ej.bluetooth.BluetoothAdapter#stopScanning()
	 */
	void onScanCompleted();

	/**
	 * Called when a advertisement or a scan response is received.
	 *
	 * @param address
	 *            the address of the device which has sent the advertisement.
	 * @param advertisementData
	 *            the advertisement data.
	 * @param rssi
	 *            the RSSI measurement.
	 * @see ej.bluetooth.BluetoothAdapter#startScanning(ej.bluetooth.BluetoothScanFilter)
	 * @see ej.bluetooth.BluetoothAdapter#stopScanning()
	 */
	void onScanResult(BluetoothAddress address, byte[] advertisementData, int rssi);

	/**
	 * Called when a connection attempt fails.
	 *
	 * @param address
	 *            the address of the device to which the connection has failed.
	 * @see ej.bluetooth.BluetoothAdapter#connect(BluetoothAddress)
	 */
	void onConnectFailed(BluetoothAddress address);

	/**
	 * Called when a connection attempt is successful or when a remote device connects to the adapter.
	 *
	 * @param connection
	 *            the connection which was established.
	 * @see ej.bluetooth.BluetoothAdapter#connect(BluetoothAddress)
	 * @see ej.bluetooth.BluetoothAdapter#startAdvertising(byte[])
	 */
	void onConnected(BluetoothConnection connection);

	/**
	 * Called when the connection with a device is closed.
	 *
	 * @param connection
	 *            the connection which was closed.
	 * @see ej.bluetooth.BluetoothConnection#disconnect()
	 */
	void onDisconnected(BluetoothConnection connection);

	/**
	 * Called when a pairing request from a device is received.
	 *
	 * @param connection
	 *            the connection with the device from which the pairing request was received.
	 * @see ej.bluetooth.BluetoothConnection#sendPairResponse(boolean)
	 */
	void onPairRequest(BluetoothConnection connection);

	/**
	 * Called when the pairing procedure with a device is completed.
	 *
	 * @param connection
	 *            the connection with the device with which the pairing procedure was completed.
	 * @param success
	 *            true if the pairing procedure was successful, false otherwise.
	 * @see ej.bluetooth.BluetoothConnection#sendPairRequest()
	 * @see ej.bluetooth.BluetoothConnection#sendPairResponse(boolean)
	 */
	void onPairCompleted(BluetoothConnection connection, boolean success);

	/**
	 * Called when a passkey request from a device is received.
	 *
	 * @param connection
	 *            the connection with the device from which the passkey request was received.
	 * @see ej.bluetooth.BluetoothConnection#sendPasskeyResponse(boolean, int)
	 */
	void onPasskeyRequest(BluetoothConnection connection);

	/**
	 * Called when a passkey is generated to be displayed to the user.
	 *
	 * @param connection
	 *            the connection with the device to which the passkey request was sent.
	 * @param passkey
	 *            the passkey which was generated.
	 * @see ej.bluetooth.BluetoothConnection#sendPasskeyResponse(boolean, int)
	 */
	void onPasskeyGenerated(BluetoothConnection connection, int passkey);

	/**
	 * Called when a service provided by a device has been discovered.
	 *
	 * @param connection
	 *            the connection with the device which provides the service.
	 * @param service
	 *            the service which has been discovered.
	 * @see ej.bluetooth.BluetoothConnection#discoverServices()
	 * @see ej.bluetooth.BluetoothConnection#discoverService(ej.bluetooth.BluetoothUuid)
	 */
	void onDiscoveryResult(BluetoothConnection connection, BluetoothService service);

	/**
	 * Called when a service discovery is completed.
	 *
	 * @param connection
	 *            the connection with the device with which the discovery was initiated.
	 * @see ej.bluetooth.BluetoothConnection#discoverServices()
	 * @see ej.bluetooth.BluetoothConnection#discoverService(ej.bluetooth.BluetoothUuid)
	 */
	void onDiscoveryCompleted(BluetoothConnection connection);
}
