/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.outline.border;

import ej.annotation.Nullable;
import ej.bon.XMath;
import ej.microui.display.GraphicsContext;
import ej.mwt.style.outline.Outline;
import ej.mwt.util.Outlineable;
import ej.mwt.util.Size;

/**
 * Represents and draw a rectangular colored border with one color.
 * <p>
 * The thicknesses are stored as a <code>char</code> for heap optimization and therefore cannot exceed
 * <code>65535</code>.
 */
public class FlexibleRectangularBorder implements Outline {

	private final int color;
	private final char top;
	private final char bottom;
	private final char left;
	private final char right;

	/**
	 * Creates a flexible rectangular border specifying its color and edges thickness.
	 * <p>
	 * The given thickness values are clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param color
	 *            the color.
	 * @param top
	 *            the top thickness.
	 * @param right
	 *            the right thickness.
	 * @param bottom
	 *            the bottom thickness.
	 * @param left
	 *            the left thickness.
	 */
	public FlexibleRectangularBorder(int color, int top, int right, int bottom, int left) {
		this.color = color;
		this.top = (char) XMath.limit(top, 0, Character.MAX_VALUE);
		this.right = (char) XMath.limit(right, 0, Character.MAX_VALUE);
		this.bottom = (char) XMath.limit(bottom, 0, Character.MAX_VALUE);
		this.left = (char) XMath.limit(left, 0, Character.MAX_VALUE);
	}

	/**
	 * Gets the color.
	 *
	 * @return the color.
	 */
	public int getColor() {
		return this.color;
	}

	/**
	 * Gets the top thickness.
	 *
	 * @return the top thickness.
	 */
	public int getTop() {
		return this.top;
	}

	/**
	 * Gets the bottom thickness.
	 *
	 * @return the bottom thickness.
	 */
	public int getBottom() {
		return this.bottom;
	}

	/**
	 * Gets the left thickness.
	 *
	 * @return the left thickness.
	 */
	public int getLeft() {
		return this.left;
	}

	/**
	 * Gets the right thickness.
	 *
	 * @return the right thickness.
	 */
	public int getRight() {
		return this.right;
	}

	@Override
	public void apply(Outlineable outlineable) {
		outlineable.removeOutline(this.left, this.top, this.right, this.bottom);
	}

	@Override
	public void apply(GraphicsContext g, Size size) {
		int width = size.getWidth();
		int height = size.getHeight();

		int top = this.top;
		int bottom = this.bottom;
		int left = this.left;
		int right = this.right;

		g.setColor(this.color);
		BorderHelper.drawRectangularBorder(g, width, height, left, top, right, bottom);

		size.removeOutline(left, top, right, bottom);

		g.translate(left, top);
		g.intersectClip(0, 0, size.getWidth(), size.getHeight());
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj instanceof FlexibleRectangularBorder) {
			FlexibleRectangularBorder border = (FlexibleRectangularBorder) obj;
			return this.color == border.color && this.top == border.top && this.bottom == border.bottom
					&& this.left == border.left && this.right == border.right;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return 15 * this.color + this.top * this.bottom * this.left * this.right;
	}

}
