/*
 * Java
 *
 * Copyright 2019-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.library.iot.rcommand.bluetooth.commands;

import java.io.IOException;

import ej.annotation.Nullable;
import ej.bluetooth.BluetoothAdapter;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.BluetoothServiceDefinition;
import ej.bluetooth.BluetoothUuid;
import ej.library.iot.rcommand.bluetooth.BluetoothController;
import ej.library.iot.rcommand.bluetooth.Commands;
import ej.rcommand.CommandReader;
import ej.rcommand.CommandSender;
import ej.rcommand.synchronous.Endpoint;

public class AddServiceEndpoint implements Endpoint {

	private static final byte ATTRIBUTE_TYPE_CHARACTERISTIC = 0;
	private static final byte ATTRIBUTE_TYPE_DESCRIPTOR = 1;

	private static final int SERVICE_HEADER_SIZE = 18;
	private static final int ATTRIBUTE_SIZE = 20;

	private final BluetoothController controller;

	private @Nullable byte[] service;

	public AddServiceEndpoint(BluetoothController controller) {
		this.controller = controller;
	}

	@Override
	public String getName() {
		return Commands.BLUETOOTH_ADD_SERVICE;
	}

	@Override
	public void readRequestBody(CommandReader reader) throws IOException {
		this.service = reader.readByteArray();
	}

	@Override
	public void writeResponseBody(CommandSender sender) throws IOException {
		byte[] service = this.service;
		assert (service != null);

		int numHandles = 1 + service[16] + service[17];
		short[] handles = new short[numHandles];

		BluetoothServiceDefinition serviceDefinition = createServiceDefinition();
		BluetoothService addedService = BluetoothAdapter.getAdapter().addService(serviceDefinition);
		if (addedService != null) {
			this.controller.onServiceAdded(addedService, handles);
		}

		byte[] handlesByteArray = new byte[numHandles * 2];
		for (int i = 0; i < numHandles; i++) {
			handlesByteArray[i * 2] = (byte) (handles[i] >> 8);
			handlesByteArray[i * 2 + 1] = (byte) handles[i];
		}

		sender.sendInt(addedService != null ? Commands.OK : Commands.ERROR);
		sender.sendByteArray(handlesByteArray, 0, numHandles * 2);
	}

	private BluetoothServiceDefinition createServiceDefinition() {
		byte[] service = this.service;
		assert (service != null);

		// read service data
		BluetoothUuid serviceUuid = new BluetoothUuid(service, 0);

		// create service definition
		BluetoothServiceDefinition definition = new BluetoothServiceDefinition(serviceUuid);

		int attrOffset = SERVICE_HEADER_SIZE;
		while (attrOffset < service.length) {
			// write attribute data
			int attrType = service[attrOffset];
			BluetoothUuid attrUuid = new BluetoothUuid(service, attrOffset + 2);
			int attrPermissions = service[attrOffset + 18];
			int attrProperties = service[attrOffset + 19];
			attrOffset += ATTRIBUTE_SIZE;

			// add attribute to service definition
			if (attrType == ATTRIBUTE_TYPE_CHARACTERISTIC) {
				definition.addCharacteristic(attrUuid, attrProperties, attrPermissions);
			} else if (attrType == ATTRIBUTE_TYPE_DESCRIPTOR) {
				definition.addDescriptor(attrUuid, attrPermissions);
			}
		}

		return definition;
	}
}
