/*
 * Java
 *
 * Copyright 2009-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.hoka.http.encoding;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import ej.hoka.http.HttpRequest;
import ej.hoka.http.HttpResponse;
import ej.hoka.io.ChunkedMessageBodyInputStream;
import ej.hoka.io.ChunkedMessageBodyOutputStream;

/**
 * HTTP-1.1 chunked transfer encoding handler to read and write data in chunked encoding.
 */
public final class ChunkedTransferCodingHandler implements TransferEncoding {

	/**
	 * Private default constructor to avoid direct instantiation.
	 */
	private ChunkedTransferCodingHandler() {
		// private constructor, because of singleton behavior
	}

	/**
	 * Private holder class for the unique instance, lazy initialized when instance is first accessed.
	 */
	private static class ChunkedTransferCodingHandlerHolder {
		/**
		 * Instance to use in factory method.
		 */
		private static final ChunkedTransferCodingHandler INSTANCE = new ChunkedTransferCodingHandler();
	}

	/**
	 * Factory method to create an instance of ChunkedTransferCodingHandler.
	 *
	 * @return an instance of {@link ChunkedTransferCodingHandler}
	 */
	public static ChunkedTransferCodingHandler getInstance() {
		return ChunkedTransferCodingHandlerHolder.INSTANCE;
	}

	// NOTES:
	// This encoding has been first encountered with AXIS2 client
	// Internal headers and extension are skipped (not used by AXIS2)
	/**
	 * Returns the internal ID of the {@link ChunkedTransferCodingHandler}.
	 *
	 * @return the String "chunked".
	 */
	@Override
	public String getId() {
		return "chunked"; //$NON-NLS-1$
	}

	/**
	 * Creates a {@link ChunkedMessageBodyInputStream} to read the body of the HTTP request in "chunked" encoding from
	 * the {@link HttpRequest} and the {@link InputStream}.
	 *
	 * @param request
	 *            the {@link HttpRequest}
	 * @param input
	 *            the {@link InputStream}
	 *
	 * @return a new instance of {@link ChunkedMessageBodyInputStream}
	 * @throws IOException
	 *             when I/O error occurs
	 */
	@Override
	public InputStream open(HttpRequest request, InputStream input) throws IOException {
		return new ChunkedMessageBodyInputStream(input);
	}

	/**
	 * Creates an {@link OutputStream} to write the body of the HTTP response in "chunked" encoding using the
	 * {@link HttpResponse} and the {@link OutputStream}.
	 *
	 * @param response
	 *            the {@link HttpResponse}.
	 * @param output
	 *            the {@link OutputStream}.
	 * @return a new instance of {@link ChunkedMessageBodyOutputStream}.
	 * @throws IOException
	 *             if an I/O error occurs.
	 */
	@Override
	public OutputStream open(HttpResponse response, OutputStream output) throws IOException {
		return new ChunkedMessageBodyOutputStream(output);
	}

}
