/*
 * Java
 *
 * Copyright 2017-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.trace;

import java.util.Iterator;
import java.util.ServiceLoader;

/**
 * Implements the natives of the library <code>ej.api#trace</code> and provides
 * the same API as the Tracer class.
 * <p>
 * The aim is to be able to add events from the native world (in addition with
 * the events added by the application and its libraries) like the application
 * does with the foundation library API.
 */
public class Tracer {

	private static TracerService tracer;

	private final int groupId;

	/**
	 * Indirection for the implementation.
	 */
	public interface TracerService {

		/**
		 * Starts to record the events.
		 * <p>
		 * By default, the trace system is stopped.
		 */
		default void startTrace() {
			// does nothing by default
		}

		/**
		 * Stops to record the events. When the record system is stopped, any call to
		 * the {@link #recordEvent(int, int, int...)} methods will not record any
		 * events.
		 * <p>
		 * By default, the trace system is stopped.
		 */
		default void stopTrace() {
			// does nothing by default
		}

		/**
		 * Tests whether the trace system is started or not.
		 *
		 * @return <code>true</code> if the trace system is started, <code>false</code>
		 *         otherwise.
		 */
		default boolean isTraceStarted() {
			return false;
		}

		/**
		 * Registers a group and returns the native identifier that identifies it.
		 *
		 * @param groupNameBuffer
		 *            name of the group. A null terminated string
		 * @param nbEventTypes
		 *            maximum number of event ids for the registered group
		 * @return a new group id for the given name
		 */
		default int registerGroup(byte[] groupNameBuffer, int nbEventTypes) {
			return 0;
		}

		/**
		 * Registers a group and returns the native identifier that identifies it.
		 *
		 * @param groupName
		 *            name of the group
		 * @param nbEventTypes
		 *            maximum number of event ids for the registered group
		 * @return a new group id for the given name
		 */
		default int registerGroup(String groupName, int nbEventTypes) {
			return 0;
		}

		/**
		 * Traces an event.
		 *
		 * @param groupId
		 *            identifier of the group of events
		 * @param eventId
		 *            the event identifier
		 * @param values
		 *            the event parameters
		 */
		default void recordEvent(int groupId, int eventId, int... values) {
			// does nothing by default
		}

		/**
		 * Traces an "end" event.
		 *
		 * @param groupId
		 *            identifier of the group of events
		 * @param eventId
		 *            the "end" event identifier
		 * @param values
		 *            the "end" event parameters
		 */
		default void recordEventEnd(int groupId, int eventId, int... values) {
			// does nothing by default
		}
	}

	private static TracerService getTracer() {
		if (null == tracer) {
			ServiceLoader<TracerService> hilLoader = ServiceLoader.load(TracerService.class,
					Tracer.class.getClassLoader());
			Iterator<TracerService> it = hilLoader.iterator();
			tracer = it.hasNext() ? it.next() :
				new TracerService() {
				// use void implementation
			};
		}
		return tracer;
	}

	/**
	 * Starts to record the events.
	 * <p>
	 * By default, the trace system is stopped.
	 */
	public static void startTrace() {
		getTracer().startTrace();
	}

	/**
	 * Stops to record the events. When the record system is stopped, any call to
	 * the {@link #recordEvent(int)} methods will not record any events.
	 * <p>
	 * By default, the trace system is stopped.
	 */
	public static void stopTrace() {
		getTracer().stopTrace();
	}

	/**
	 * Tests whether the trace system is started or not.
	 *
	 * @return <code>true</code> if the trace system is started, <code>false</code>
	 *         otherwise.
	 */
	public static boolean isTraceStarted() {
		return getTracer().isTraceStarted();
	}

	/**
	 * Register a group and returns the native identifier that identifies it.
	 *
	 * @param groupNameBuffer
	 *            name of the group. A null terminated string
	 * @param nbEventTypes
	 *            maximum number of event ids for the registered group
	 * @return a new group id for the given name
	 */
	public static int registerGroup(byte[] groupNameBuffer, int nbEventTypes) {
		return getTracer().registerGroup(groupNameBuffer, nbEventTypes);
	}

	/**
	 * Records an event.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 */
	public static void recordEventNative(int groupId, int eventId) {
		getTracer().recordEvent(groupId, eventId);
	}

	/**
	 * Records an event with one custom value.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1) {
		getTracer().recordEvent(groupId, eventId, value1);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2) {
		getTracer().recordEvent(groupId, eventId, value1, value2);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2, int value3) {
		getTracer().recordEvent(groupId, eventId, value1, value2, value3);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2, int value3, int value4) {
		getTracer().recordEvent(groupId, eventId, value1, value2, value3, value4);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2, int value3, int value4,
			int value5) {
		getTracer().recordEvent(groupId, eventId, value1, value2, value3, value4, value5);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2, int value3, int value4,
			int value5, int value6) {
		getTracer().recordEvent(groupId, eventId, value1, value2, value3, value4, value5, value6);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 * @param value7
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2, int value3, int value4,
			int value5, int value6, int value7) {
		getTracer().recordEvent(groupId, eventId, value1, value2, value3, value4, value5, value6, value7);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 * @param value7
	 *            custom value for the event
	 * @param value8
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2, int value3, int value4,
			int value5, int value6, int value7, int value8) {
		getTracer().recordEvent(groupId, eventId, value1, value2, value3, value4, value5, value6, value7, value8);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 * @param value7
	 *            custom value for the event
	 * @param value8
	 *            custom value for the event
	 * @param value9
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2, int value3, int value4,
			int value5, int value6, int value7, int value8, int value9) {
		getTracer().recordEvent(groupId, eventId, value1, value2, value3, value4, value5, value6, value7, value8,
				value9);
	}

	/**
	 * Records an event with several custom values.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 * @param value7
	 *            custom value for the event
	 * @param value8
	 *            custom value for the event
	 * @param value9
	 *            custom value for the event
	 * @param value10
	 *            custom value for the event
	 */
	public static void recordEventNative(int groupId, int eventId, int value1, int value2, int value3, int value4,
			int value5, int value6, int value7, int value8, int value9, int value10) {
		getTracer().recordEvent(groupId, eventId, value1, value2, value3, value4, value5, value6, value7, value8,
				value9, value10);
	}

	/**
	 * Record the end of the execution of an event.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 */
	public static void recordEventEndNative(int groupId, int eventId) {
		getTracer().recordEventEnd(groupId, eventId);
	}

	/**
	 * Record the end of the execution of an event with one custom value.
	 *
	 * @param groupId
	 *            identifier of the group of events
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 */
	public static void recordEventEndNative(int groupId, int eventId, int value1) {
		getTracer().recordEventEnd(groupId, eventId, value1);
	}

	/**
	 * Creates a new tracer with the given name. The created {@link Tracer} defines
	 * <code>nbEventTypes</code> types of events. The event identifiers available
	 * for this group are from 0 to <code>nbEventTypes</code>-1.
	 *
	 * @param name
	 *            name that identifies the group. Must not be <code>null</code>
	 * @param nbEventTypes
	 *            maximum number of event types available for the group
	 */
	public Tracer(String name, int nbEventTypes) throws InternalError {
		this.groupId = getTracer().registerGroup(name, nbEventTypes);
	}

	/**
	 * Gets the group identifier. This identifier is unique for each {@link Tracer}
	 * instance.
	 *
	 * @return the group unique identifier.
	 */
	public int getGroupID() {
		return this.groupId;
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 */
	public void recordEvent(int eventId) {
		recordEventNative(this.groupId, eventId);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1) {
		recordEventNative(this.groupId, eventId, value1);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2) {
		recordEventNative(this.groupId, eventId, value1, value2);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2, int value3) {
		recordEventNative(this.groupId, eventId, value1, value2, value3);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2, int value3, int value4) {
		recordEventNative(this.groupId, eventId, value1, value2, value3, value4);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2, int value3, int value4, int value5) {
		recordEventNative(this.groupId, eventId, value1, value2, value3, value4, value5);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2, int value3, int value4, int value5, int value6) {
		recordEventNative(this.groupId, eventId, value1, value2, value3, value4, value5, value6);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 * @param value7
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2, int value3, int value4, int value5, int value6,
			int value7) {
		recordEventNative(this.groupId, eventId, value1, value2, value3, value4, value5, value6, value7);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 * @param value7
	 *            custom value for the event
	 * @param value8
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2, int value3, int value4, int value5, int value6,
			int value7, int value8) {
		recordEventNative(this.groupId, eventId, value1, value2, value3, value4, value5, value6, value7, value8);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 * @param value7
	 *            custom value for the event
	 * @param value8
	 *            custom value for the event
	 * @param value9
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2, int value3, int value4, int value5, int value6,
			int value7, int value8, int value9) {
		recordEventNative(this.groupId, eventId, value1, value2, value3, value4, value5, value6, value7, value8,
				value9);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event
	 * @param value2
	 *            custom value for the event
	 * @param value3
	 *            custom value for the event
	 * @param value4
	 *            custom value for the event
	 * @param value5
	 *            custom value for the event
	 * @param value6
	 *            custom value for the event
	 * @param value7
	 *            custom value for the event
	 * @param value8
	 *            custom value for the event
	 * @param value9
	 *            custom value for the event
	 * @param value10
	 *            custom value for the event
	 */
	public void recordEvent(int eventId, int value1, int value2, int value3, int value4, int value5, int value6,
			int value7, int value8, int value9, int value10) {
		recordEventNative(this.groupId, eventId, value1, value2, value3, value4, value5, value6, value7, value8, value9,
				value10);
	}

	/**
	 * Records an event for this {@link Tracer}. The given event identifier must be
	 * valid for this {@link Tracer} (i.e. between 0 and <code>nbEventTypes</code>-1
	 * included where <code>nbEventTypes</code> is the value given to the
	 * {@link #Tracer(String, int)} constructor).
	 * <p>
	 * The record is done only if the trace system is started (see
	 * <code>{@link #startTrace()}</code>).
	 *
	 * @param eventId
	 *            the event identifier
	 * @param values
	 *            the event parameters
	 */
	public void recordEvent(int eventId, int... values) {
		getTracer().recordEvent(getGroupID(), eventId, values);
	}

	/**
	 * Records the end of the execution of an event for this {@link Tracer}. Call
	 * this method to trace the duration of an event previously record with one of
	 * the {@link #recordEvent(int)} methods. This method should not be called for
	 * event that has no duration.
	 * <p>
	 * For example, if you want to trace the execution of a method, you can call
	 * {@link #recordEvent(int)} at the beginning of the function and
	 * {@link #recordEventEnd(int)} at the end of the function.
	 * <p>
	 * The record is done only if the trace system is started (see
	 * {@link #startTrace()}).
	 *
	 * @param eventId
	 *            identifier of the event
	 */
	public void recordEventEnd(int eventId) {
		recordEventEndNative(this.groupId, eventId);
	}

	/**
	 * Records the end of the execution of an event for this {@link Tracer}. Call
	 * this method to trace the duration of an event previously record with one of
	 * the {@link #recordEvent(int)} methods. This method should not be called for
	 * event that has no duration.
	 * <p>
	 * For example, if you want to trace the execution of a method, you can call
	 * {@link #recordEvent(int)} at the beginning of the function and
	 * {@link #recordEventEnd(int, int)} at the end of the function.
	 * <p>
	 * The record is done only if the trace system is started (see
	 * {@link #startTrace()}).
	 *
	 * @param eventId
	 *            identifier of the event
	 * @param value1
	 *            custom value for the event end.
	 */
	public void recordEventEnd(int eventId, int value1) {
		recordEventEndNative(this.groupId, eventId, value1);
	}

	/**
	 * Records the end of the execution of an event for this {@link Tracer}. Call
	 * this method to trace the duration of an event previously record with one of
	 * the {@link #recordEvent(int)} methods. This method should not be called for
	 * event that has no duration.
	 * <p>
	 * For example, if you want to trace the execution of a method, you can call
	 * {@link #recordEvent(int)} at the beginning of the function and
	 * {@link #recordEventEnd(int)} at the end of the function.
	 * <p>
	 * The record is done only if the trace system is started (see
	 * {@link #startTrace()}).
	 *
	 * @param eventId
	 *            the "end" event identifier
	 * @param values
	 *            the "end" event parameters
	 */
	public void recordEventEnd(int eventId, int... values) {
		getTracer().recordEventEnd(getGroupID(), eventId, values);
	}

}
