/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.bon.Constants;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Tests the drawing of strings on circle.
 */
public class TestDrawStringOnCircle {

	private static final String STRING_SQUARE = "A";
	private static final String STRING_SQUARE_BASELINE_SQUARE = "ABA";
	private static final String STRING_ARC = "FGGFFF";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test drawing a string with different colors.
	 */
	@Test
	public void testStringOnCircleColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_ARC;
		float fontSize = 30;
		float radius = 150;
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);

		// set the center of the circle
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			TestUtilities.drawStringOnCircleGrid(font, fontSize, radius, x, y, Direction.CLOCKWISE);
		}

		int[] colorSet = { Colors.WHITE, Colors.CYAN, Colors.RED, Colors.GREEN, Colors.MAGENTA };
		for (int color : colorSet) {
			g.setColor(color);

			VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
					GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
			display.flush();

			// target a pixel in each character for the test
			float currentAngle = 0;
			for (int index = 0; index < string.length(); index++) {
				// compute glyph position
				char character = string.charAt(index);
				float charWidth = TestUtilities.measureCharWidth("" + character, font, fontSize);
				// angle offset is negative in clockwise direction
				float charAngle = -charWidth / radius;
				float angleToCharCenter = currentAngle + charAngle / 2;
				float targetToCenter = topCircleRadius - fontSize / 10;
				int targetX = (int) (x + targetToCenter * Math.cos(angleToCharCenter));
				int targetY = (int) (y - targetToCenter * Math.sin(angleToCharCenter));

				TestUtilities.check("color #" + Integer.toHexString(color), targetX, targetY, g, color);

				currentAngle = currentAngle + charAngle;
			}
		}
	}

	/**
	 * Test drawing a string with clockwise direction.
	 */
	@Test
	public void testStringOnCircleCounterClockwise() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_ARC;
		float fontSize = 30;
		float radius = 150;
		float topCircleRadius = radius - font.getBaselinePosition(fontSize);
		int color = Colors.WHITE;

		// set the center of the circle
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			TestUtilities.drawStringOnCircleGrid(font, fontSize, radius, x, y, Direction.COUNTER_CLOCKWISE);
		}

		g.setColor(color);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.COUNTER_CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		display.flush();

		// target a pixel in each character for the test
		float currentAngle = 0;
		for (int index = 0; index < string.length(); index++) {
			// compute glyph position
			char character = string.charAt(index);
			float charWidth = TestUtilities.measureCharWidth("" + character, font, fontSize);
			// angle offset is positive in counter-clockwise direction
			float charAngle = charWidth / radius;
			float angleToCharCenter = currentAngle + charAngle / 2;
			float targetToCenter = topCircleRadius + fontSize / 10;
			int targetX = (int) (x + targetToCenter * Math.cos(angleToCharCenter));
			int targetY = (int) (y - targetToCenter * Math.sin(angleToCharCenter));

			TestUtilities.check("direction - CCW", targetX, targetY, g, color);

			currentAngle = currentAngle + charAngle;
		}
	}

	/**
	 * Test drawing a string with different rotation angles.
	 */
	@Test
	public void testStringOnCircleAngle() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_ARC;
		float fontSize = 24;
		float radius = 150;
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		int color = Colors.WHITE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		for (float angle = 0; angle < 270; angle += 45) {
			// set the center of the circle and rotation
			Matrix matrix = new Matrix();
			matrix.setTranslate(x, y);
			matrix.preRotate(angle);

			g.setColor(color);
			VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
					GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
			display.flush();

			// target a pixel in each character for the test
			float currentAngle = -(float) Math.toRadians(angle);
			for (int index = 0; index < string.length(); index++) {
				// compute glyph position
				char character = string.charAt(index);
				float charWidth = TestUtilities.measureCharWidth("" + character, font, fontSize);
				// angle offset is negative in clockwise direction
				float charAngle = -charWidth / radius;
				float angleToCharCenter = currentAngle + charAngle / 2;

				float targetToCenter = topCircleRadius - fontSize / 10;
				int targetX = (int) (x + targetToCenter * Math.cos(angleToCharCenter));
				int targetY = (int) (y - targetToCenter * Math.sin(angleToCharCenter));

				TestUtilities.check("angle " + angle, targetX, targetY, g, color);

				currentAngle = currentAngle + charAngle;
			}

			TestUtilities.clearScreen();
		}
	}

	/**
	 * Test drawing a string on circle with a clipping area.
	 */
	@Test
	public void testStringOnCircleClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;

		float fontSize = 24;
		float radius = 150;
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		int color = Colors.WHITE;
		float charWidth = TestUtilities.measureCharWidth(string.substring(0, 1), font, fontSize);

		// set the center of the circle
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		// set the clip to be on the first character only
		int clipX = (int) (x + radius);
		int clipY = (int) (y + fontSize / 10);
		int clipWidth = (int) (fontSize / 2);
		int clipHeight = (int) (charWidth / 2);
		g.setClip(clipX, clipY, clipWidth, clipHeight);

		g.setColor(color);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		display.flush();

		// compute glyph position
		// angle offset is negative in clockwise direction
		float charAngle = -charWidth / radius;
		float angleToCharCenter = charAngle / 2;

		float targetToCenter = topCircleRadius - fontSize / 2;
		int targetX = (int) (x + targetToCenter * Math.cos(angleToCharCenter));
		int targetY = (int) (y - targetToCenter * Math.sin(angleToCharCenter));

		// check that inside the first character (in the clip area)
		TestUtilities.check("clip", targetX, targetY, g, color);
		// check the area outside the clip area
		TestUtilities.checkPeripheralArea("clip", TestUtilities.BACKGROUND_COLOR, clipX, clipY, clipWidth, clipHeight,
				50, 3);

		g.resetClip();

	}

	/**
	 * Test drawing a gradient string on circle with a clipping area.
	 */
	@Test
	public void testStringOnCircleGradientClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;

		float fontSize = 24;
		float radius = 150;
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		int color = Colors.WHITE;
		float charWidth = TestUtilities.measureCharWidth(string.substring(0, 1), font, fontSize);

		// set the center of the circle
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		// set the clip to be on the first character only
		int clipX = (int) (x + radius);
		int clipY = (int) (y + fontSize / 10);
		int clipWidth = (int) (fontSize / 2);
		int clipHeight = (int) (charWidth / 2);
		g.setClip(clipX, clipY, clipWidth, clipHeight);

		// gradient definition
		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0, 1 };
		LinearGradient gradient = new LinearGradient(0, 0, 10, 0, colors, stops);

		g.setColor(color);
		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		display.flush();

		// compute glyph position
		// angle offset is negative in clockwise direction
		float charAngle = -charWidth / radius;
		float angleToCharCenter = charAngle / 2;

		float targetToCenter = topCircleRadius - fontSize / 2;
		int targetX = (int) (x + targetToCenter * Math.cos(angleToCharCenter));
		int targetY = (int) (y - targetToCenter * Math.sin(angleToCharCenter));

		// check that inside the first character (in the clip area)
		TestUtilities.checkNot("clip", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);
		// check the area outside the clip area
		TestUtilities.checkPeripheralArea("clip", TestUtilities.BACKGROUND_COLOR, clipX, clipY, clipWidth, clipHeight,
				50, 3);

		g.resetClip();
	}

	/**
	 * Test drawing a string with different alpha. The drawn character is a rectangle filling the whole character box.
	 */
	@Test
	public void testStringOnCircleAlpha() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;
		float fontSize = 24;
		float radius = 150;
		int color = Colors.WHITE;
		float charWidth = TestUtilities.measureCharWidth(string.substring(0, 1), font, fontSize);

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		int[] alphaSet = { 0x0, 0x40, 0x80, 0xC0, 0xFF };
		for (int alpha : alphaSet) {
			g.setColor(color);
			VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
					alpha, BlendMode.SRC_OVER, 0);
			display.flush();

			// Check the color at the center of the char
			// angle offset is negative in clockwise direction
			float centerOfGlyphToCircleCenter = radius + font.getBaselinePosition(fontSize) - fontSize / 2;
			float charAngle = -charWidth / centerOfGlyphToCircleCenter;
			float angleToCharCenter = charAngle / 2;

			int targetX = (int) (x + centerOfGlyphToCircleCenter * Math.cos(angleToCharCenter));
			int targetY = (int) (y - centerOfGlyphToCircleCenter * Math.sin(angleToCharCenter));

			int expectedColor = TestUtilities.blendColors(color,
					display.getDisplayColor(TestUtilities.BACKGROUND_COLOR), alpha);
			TestUtilities.check("alpha", targetX, targetY, g, expectedColor);

			if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
				g.setColor(Colors.RED);
				Painter.drawRectangle(g, targetX, targetY, 3, 3);
				display.flush();
			}

			TestUtilities.clearScreen();
		}
	}

	/**
	 * Test drawing a string with different alpha. The drawn character is a rectangle filling the whole character box.
	 */
	@Test
	public void testStringOnCircleAlphaCounterClockwise() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE_BASELINE_SQUARE;
		float fontSize = 24;
		float radius = 150;
		int color = Colors.WHITE;
		float charWidth = TestUtilities.measureCharWidth(string.substring(0, 1), font, fontSize);

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		int[] alphaSet = { 0x0, 0x40, 0x80, 0xC0, 0xFF };
		for (int alpha : alphaSet) {
			g.setColor(color);
			VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius,
					Direction.COUNTER_CLOCKWISE, alpha, BlendMode.SRC_OVER, 0);
			display.flush();

			// Check the color at the center of the char
			// angle offset is positive in counter clockwise direction
			float centerOfGlyphToCircleCenter = radius - font.getBaselinePosition(fontSize) + fontSize / 2;
			float charAngle = charWidth / centerOfGlyphToCircleCenter;
			float angleToCharCenter = charAngle / 2;

			int targetX = (int) (x + centerOfGlyphToCircleCenter * Math.cos(angleToCharCenter));
			int targetY = (int) (y - centerOfGlyphToCircleCenter * Math.sin(angleToCharCenter));

			int expectedColor = TestUtilities.blendColors(color,
					display.getDisplayColor(TestUtilities.BACKGROUND_COLOR), alpha);
			TestUtilities.check("alpha", targetX, targetY, g, expectedColor);

			if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
				g.setColor(Colors.RED);
				Painter.drawRectangle(g, targetX, targetY, 3, 3);
				display.flush();
			}

			TestUtilities.clearScreen();
		}
	}

	/**
	 * Test drawing a string on circle with a gradient. The drawn character is a rectangle filling the whole character
	 * box.
	 */
	@Test
	public void testStringOnCircleGradient() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		float fontSize = 48;
		float radius = 150;
		String string = STRING_SQUARE_BASELINE_SQUARE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		float stringWidth = TestUtilities.measureCharWidth(string, font, fontSize);
		float bottomOfStringToCircleCenter = radius + font.getBaselinePosition(fontSize) - fontSize;
		float centerOfStringToCircleCenter = bottomOfStringToCircleCenter + fontSize / 2;
		// angle offset is negative in clockwise direction
		float stringAngle = -stringWidth / centerOfStringToCircleCenter;

		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0.2f, 0.8f };

		float gradientEndX = (float) (bottomOfStringToCircleCenter * Math.cos(stringAngle));
		float gradientEndY = (float) (-bottomOfStringToCircleCenter * Math.sin(stringAngle));

		LinearGradient gradient = new LinearGradient(bottomOfStringToCircleCenter, 0, gradientEndX, gradientEndY,
				colors, stops);

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		display.flush();

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.YELLOW);
			Painter.drawRectangle(g, (int) gradientEndX + x, (int) gradientEndY + y, 1, 1);
			Painter.drawRectangle(g, (int) (x + bottomOfStringToCircleCenter), y, 1, 1);
		}

		// Check the color at the first character
		double angleAtFirstChar = stringAngle * 0.05f;
		float targetX = (int) (x + radius * Math.cos(angleAtFirstChar));
		float targetY = (int) (y - radius * Math.sin(angleAtFirstChar));

		TestUtilities.check("gradient start CW", (int) targetX, (int) targetY, g, Colors.BLUE);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 3, 3);
		}

		// Check color at the last character
		double angleAtLastChar = stringAngle * 0.90f;
		targetX = (int) (x + radius * Math.cos(angleAtLastChar));
		targetY = (int) (y - radius * Math.sin(angleAtLastChar));

		TestUtilities.check("gradient end CW", (int) targetX, (int) targetY, g, Colors.GREEN);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 3, 3);
			display.flush();
		}

	}

	/**
	 * Test drawing a string on circle with a gradient in counterclockwise direction. The drawn character is a rectangle
	 * filling the whole character box.
	 */
	@Test
	public void testStringOnCircleGradientCounterClockwise() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		float fontSize = 48;
		float radius = 150;
		String string = STRING_SQUARE_BASELINE_SQUARE;

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		float stringWidth = TestUtilities.measureCharWidth(string, font, fontSize);
		float bottomOfStringToCircleCenter = radius - font.getBaselinePosition(fontSize);
		float centerOfStringToCircleCenter = bottomOfStringToCircleCenter + fontSize / 2;
		// angle offset is positive in counter-clockwise direction
		float stringAngle = stringWidth / centerOfStringToCircleCenter;

		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0, 1 };

		float gradientEndX = (float) (bottomOfStringToCircleCenter * Math.cos(stringAngle));
		float gradientEndY = (float) (-bottomOfStringToCircleCenter * Math.sin(stringAngle));

		LinearGradient gradient = new LinearGradient(bottomOfStringToCircleCenter, 0, gradientEndX, gradientEndY,
				colors, stops);

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.COUNTER_CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		display.flush();

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.YELLOW);
			Painter.drawRectangle(g, (int) gradientEndX + x, (int) gradientEndY + y, 1, 1);
			Painter.drawRectangle(g, (int) (x + bottomOfStringToCircleCenter), y, 1, 1);
		}

		// Check the color at the first character
		double angleAtFirstChar = stringAngle * 0.05f;
		float targetX = (int) (x + radius * Math.cos(angleAtFirstChar));
		float targetY = (int) (y - radius * Math.sin(angleAtFirstChar));

		TestUtilities.check("gradient start CCW", (int) targetX, (int) targetY, g, Colors.BLUE);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 3, 3);
		}

		// Check color at the last character
		double angleAtLastChar = stringAngle * 0.90f;
		targetX = (int) (x + radius * Math.cos(angleAtLastChar));
		targetY = (int) (y - radius * Math.sin(angleAtLastChar));

		TestUtilities.check("gradient end CCW", (int) targetX, (int) targetY, g, Colors.GREEN);

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 3, 3);
			display.flush();
		}
	}

	/**
	 * Run all tests in a row with a translation vector applied on the graphics context.
	 */
	@Test
	public void testStringOnCircleGCTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		// Translate graphic context origin
		g.translate(9, 25);

		testStringOnCircleAlpha();
		TestUtilities.clearScreen();

		testStringOnCircleAlphaCounterClockwise();
		TestUtilities.clearScreen();

		testStringOnCircleAngle();
		TestUtilities.clearScreen();

		testStringOnCircleClip();
		TestUtilities.clearScreen();

		testStringOnCircleColor();
		TestUtilities.clearScreen();

		testStringOnCircleCounterClockwise();
		TestUtilities.clearScreen();

		testStringOnCircleGradient();
		TestUtilities.clearScreen();

		testStringOnCircleGradientCounterClockwise();
		TestUtilities.clearScreen();

		g.resetTranslation();
	}

	private static VectorFont getTestFont() {
		return VectorFont.loadFont("/fonts/firstfont.ttf");
	}

	/**
	 * Test that calling the drawStringOnCircle methods on a closed font throws an exception.
	 */
	@Test
	public void testStringOnCircleFontIsClosed() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorFont font = getTestFont();
		Matrix matrix = new Matrix();

		font.close();

		int errorCode = Integer.MAX_VALUE;
		int radius = 100;
		try {
			VectorGraphicsPainter.drawStringOnCircle(g, STRING_ARC, font, 30, matrix, radius, Direction.CLOCKWISE);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("drawStringOnCircle on closed font", errorCode, VectorGraphicsException.RESOURCE_CLOSED);

		errorCode = Integer.MAX_VALUE;
		try {
			VectorGraphicsPainter.drawStringOnCircle(g, STRING_ARC, font, 30, matrix, radius, Direction.CLOCKWISE,
					GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("drawStringOnCircle full on closed font", errorCode,
				VectorGraphicsException.RESOURCE_CLOSED);

		int[] colors = { Colors.BLUE, Colors.GREEN };
		float[] positions = { 0x0, 0xFF };
		LinearGradient gradient = new LinearGradient(0, 0, 1, 1, colors, positions);

		errorCode = Integer.MAX_VALUE;
		try {
			VectorGraphicsPainter.drawGradientStringOnCircle(g, STRING_ARC, font, 30, matrix, gradient, radius,
					Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("drawGradientStringOnCircle with gradient on closed font", errorCode,
				VectorGraphicsException.RESOURCE_CLOSED);
	}

	/**
	 * Test drawing a string with an alpha out of range.
	 */
	@Test
	public void testStringOnCircleAlphaLimits() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE;
		float fontSize = 30;
		float radius = 150;
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		int color = Colors.WHITE;

		// set the center of the circle
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(color);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE, -1,
				BlendMode.SRC_OVER, 0);

		float charWidth = TestUtilities.measureCharWidth(string, font, fontSize);
		float charAngle = -charWidth / radius;
		float angleToCharCenter = charAngle / 2;
		float targetToCenter = topCircleRadius - fontSize / 2;
		int targetX = (int) (x + targetToCenter * Math.cos(angleToCharCenter));
		int targetY = (int) (y - targetToCenter * Math.sin(angleToCharCenter));

		// check that nothing has been drawn
		TestUtilities.check("alpha < 0", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);

		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE, 256,
				BlendMode.SRC_OVER, 0);
		display.flush();

		// check that nothing has been drawn
		TestUtilities.check("alpha > 255", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);

		int[] colors = { Colors.BLUE, Colors.GREEN };
		float[] positions = { 0x0, 0xFF };
		LinearGradient gradient = new LinearGradient(0, 0, 1, 1, colors, positions);

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.CLOCKWISE, -1, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.check("gradient alpha < 0", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.CLOCKWISE, 256, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.check("gradient alpha > 255", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);
	}

	/**
	 * Test drawing a string with a size <= 0.
	 */
	@Test
	public void testStringCircleSizeLimits() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE;
		float radius = 150;
		int color = Colors.WHITE;

		// set the center of the circle
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(color);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, 0, matrix, radius, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.checkArea("size = 0", TestUtilities.BACKGROUND_COLOR, 0, 0, display.getWidth(),
				display.getHeight());

		VectorGraphicsPainter.drawStringOnCircle(g, string, font, -1, matrix, radius, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.checkArea("size = 0", TestUtilities.BACKGROUND_COLOR, 0, 0, display.getWidth(),
				display.getHeight());

		int[] colors = { Colors.BLUE, Colors.GREEN };
		float[] positions = { 0x0, 0xFF };
		LinearGradient gradient = new LinearGradient(0, 0, 1, 1, colors, positions);

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, 0, matrix, gradient, radius,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.checkArea("gradient size = 0", TestUtilities.BACKGROUND_COLOR, 0, 0, display.getWidth(),
				display.getHeight());

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, -1, matrix, gradient, radius,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.checkArea("gradient size < 0", TestUtilities.BACKGROUND_COLOR, 0, 0, display.getWidth(),
				display.getHeight());
	}

	/**
	 * Test drawing a string with a radius <= 0.
	 */
	@Test
	public void testStringOnCircleRadiusLimits() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SQUARE;
		float fontSize = 30;
		int color = Colors.WHITE;

		// set the center of the circle
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(color);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, 0, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.checkArea("radius = 0", TestUtilities.BACKGROUND_COLOR, 0, 0, display.getWidth(),
				display.getHeight());

		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, -1, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.checkArea("radius < 0", TestUtilities.BACKGROUND_COLOR, 0, 0, display.getWidth(),
				display.getHeight());

		int[] colors = { Colors.BLUE, Colors.GREEN };
		float[] positions = { 0x0, 0xFF };
		LinearGradient gradient = new LinearGradient(0, 0, 1, 1, colors, positions);

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, 0,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.checkArea("gradient radius = 0", TestUtilities.BACKGROUND_COLOR, 0, 0, display.getWidth(),
				display.getHeight());

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, -1,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// check that nothing has been drawn
		TestUtilities.checkArea("gradient radius < 0", TestUtilities.BACKGROUND_COLOR, 0, 0, display.getWidth(),
				display.getHeight());
	}

	/**
	 * Test drawing a string with positive letter spacing.
	 */
	@Test
	public void testStringOnCirclePositiveLetterSpacing() {

	}

	/**
	 * Tests that drawing an empty string does nothing.
	 */
	@Test
	public void testEmptyString() {
		Display display = Display.getDisplay();

		final int fontSize = 24;
		final float radius = 50;
		final int x = display.getWidth() / 2;
		final int y = display.getHeight() / 2;

		// Place rectangle glyph at the bottom of the circle.
		final float angle = (float) (90 - Math.toDegrees(fontSize / (2 * radius)));
		GraphicsContext g = display.getGraphicsContext();

		// Ensure flush bounds are reset.
		display.flush();

		// Draw an empty string.
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		matrix.preRotate(angle);

		VectorFont font = getTestFont();
		String string = ""; //$NON-NLS-1$
		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// Display the results.
		display.flush();

		TestUtilities.checkArea("Empty string (standard): not drawn", TestUtilities.BACKGROUND_COLOR, 0, 0, //$NON-NLS-1$
				display.getWidth(), display.getHeight());
	}

	/**
	 * Tests that drawing an empty string does nothing.
	 */
	@Test
	public void testEmptyGradientString() {
		Display display = Display.getDisplay();

		final int fontSize = 24;
		final float radius = 50;
		final int x = display.getWidth() / 2;
		final int y = display.getHeight() / 2;

		// Place rectangle glyph at the bottom of the circle.
		final float angle = (float) (90 - Math.toDegrees(fontSize / (2 * radius)));
		GraphicsContext g = display.getGraphicsContext();

		// Ensure flush bounds are reset.
		display.flush();

		// Draw an empty string.
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		matrix.preRotate(angle);

		VectorFont font = getTestFont();
		String string = ""; //$NON-NLS-1$
		LinearGradient gradient = new LinearGradient(0, 0, g.getWidth(), g.getHeight(),
				new int[] { Colors.RED, Colors.BLUE });
		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// Display the results.
		display.flush();

		// Check that the flush bounds are left unchanged and nothing has been drawn.
		TestUtilities.checkArea("Empty string (gradient): not drawn", TestUtilities.BACKGROUND_COLOR, 0, 0, //$NON-NLS-1$
				display.getWidth(), display.getHeight());
	}

	/**
	 * Tests that drawing a string outside of the clip area does nothing.
	 */
	@Test
	public void testEmptyStringOutOfClip() {
		Display display = Display.getDisplay();

		final int fontSize = 24;
		final float radius = 50;
		final int x = display.getWidth() / 3;
		final int y = display.getHeight() / 3;

		// Place rectangle glyph at the bottom of the circle.
		final float angle = (float) (90 - Math.toDegrees(fontSize / (2 * radius)));
		GraphicsContext g = display.getGraphicsContext();

		// Ensure flush bounds are reset.
		display.flush();

		// Set the clip.
		g.setClip(2 * x, 2 * y, x, y);

		// Draw outside of the clip area.
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		matrix.preRotate(angle);

		VectorFont font = getTestFont();
		String string = STRING_SQUARE;
		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// Display the results.
		display.flush();

		TestUtilities.checkArea("Out-of-clip string (standard): not drawn", TestUtilities.BACKGROUND_COLOR, 0, 0, //$NON-NLS-1$
				display.getWidth(), display.getHeight());
	}

	/**
	 * Tests that drawing a string outside of the clip area does nothing.
	 */
	@Test
	public void testEmptyGradientStringOutOfClip() {
		Display display = Display.getDisplay();

		final int fontSize = 24;
		final float radius = 50;
		final int x = display.getWidth() / 3;
		final int y = display.getHeight() / 3;

		// Place rectangle glyph at the bottom of the circle.
		final float angle = (float) (90 - Math.toDegrees(fontSize / (2 * radius)));
		GraphicsContext g = display.getGraphicsContext();

		// Ensure flush bounds are reset.
		display.flush();

		// Set the clip.
		g.setClip(2 * x, 2 * y, x, y);

		// Draw outside of the clip area.
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		matrix.preRotate(angle);

		VectorFont font = getTestFont();
		String string = STRING_SQUARE;
		LinearGradient gradient = new LinearGradient(0, 0, g.getWidth(), g.getHeight(),
				new int[] { Colors.RED, Colors.BLUE });
		g.setColor(Colors.WHITE);
		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		// Display the results.
		display.flush();

		TestUtilities.checkArea("Out-of-clip string (gradient): not drawn", TestUtilities.BACKGROUND_COLOR, 0, 0, //$NON-NLS-1$
				display.getWidth(), display.getHeight());
	}
}
