/*
 * Copyright (c) 2012, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2023 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * This file is available under and governed by the GNU General Public
 * License version 2 only, as published by the Free Software Foundation.
 * However, the following notice accompanied the original version of this
 * file:
 *
 * Copyright (c) 2012, Stephen Colebourne & Michael Nascimento Santos
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  * Neither the name of JSR-310 nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package java.time.temporal;

import java.time.DateTimeException;

import ej.annotation.Nullable;

/**
 * Strategy for querying a temporal object.
 * <p>
 * Queries are a key tool for extracting information from temporal objects. They exist to externalize the process of
 * querying, permitting different approaches, as per the strategy design pattern. Examples might be a query that checks
 * if the date is the day before February 29th in a leap year, or calculates the number of days to your next birthday.
 * <p>
 * The {@link TemporalField} interface provides another mechanism for querying temporal objects. That interface is
 * limited to returning a {@code long}. By contrast, queries can return any type.
 * <p>
 * There are two equivalent ways of using a {@code TemporalQuery}. The first is to invoke the method on this interface
 * directly. The second is to use {@link TemporalAccessor#query(TemporalQuery)}:
 *
 * <pre>
 * // these two lines are equivalent, but the second approach is recommended
 * temporal = thisQuery.queryFrom(temporal);
 * temporal = temporal.query(thisQuery);
 * </pre>
 *
 * It is recommended to use the second approach, {@code query(TemporalQuery)}, as it is a lot clearer to read in code.
 * <p>
 * The most common implementations are method references, such as {@code LocalDate::from} and {@code ZoneId::from}.
 * Additional common queries are provided as static methods in {@link TemporalQueries}.
 * <p> This interface places no restrictions on the mutability of implementations, however immutability is
 *           strongly recommended.
 *
 * @param <R>
 *            the type returned from the query
 */
public interface TemporalQuery<R> {

	/**
	 * Queries the specified temporal object.
	 * <p>
	 * This queries the specified temporal object to return an object using the logic encapsulated in the implementing
	 * class. Examples might be a query that checks if the date is the day before February 29th in a leap year, or
	 * calculates the number of days to your next birthday.
	 * <p>
	 * There are two equivalent ways of using this method. The first is to invoke this method directly. The second is to
	 * use {@link TemporalAccessor#query(TemporalQuery)}:
	 *
	 * <pre>
	 * // these two lines are equivalent, but the second approach is recommended
	 * temporal = thisQuery.queryFrom(temporal);
	 * temporal = temporal.query(thisQuery);
	 * </pre>
	 *
	 * It is recommended to use the second approach, {@code query(TemporalQuery)}, as it is a lot clearer to read in
	 * code.
	 * <p>
	 *     The implementation must take the input object and query it. The implementation defines the logic of the
	 *           query and is responsible for documenting that logic. It may use any method on {@code TemporalAccessor}
	 *           to determine the result. The input object must not be altered.
	 *           <p>
	 *           The input temporal object may be in a calendar system other than ISO. Implementations may choose to
	 *           document compatibility with other calendar systems, or reject non-ISO temporal objects by
	 *           {@link TemporalQueries#chronology() querying the chronology}.
	 *           <p>
	 *           This method may be called from multiple threads in parallel. It must be thread-safe when invoked.
	 *
	 * @param temporal
	 *            the temporal object to query, not null
	 * @return the queried value, may return null to indicate not found
	 * @throws DateTimeException
	 *             if unable to query
     *
	 */
	@Nullable R queryFrom(TemporalAccessor temporal);

}
