/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.monitoring;

import ej.basictool.map.PackedMap;
import ej.kf.Module;

/**
 * This class represents the data structure used to handle data for resource monitoring.
 */
public class MonitoringData {
	private final PackedMap<Module, ExecutionCounter> executionCounterPerModuleMap;
	private final PackedMap<Module, Long> ramUsagePerModuleMap;
	private final PackedMap<Module, Integer> originalQuotaPerModuleMap;
	private final boolean runGcBeforeCollecting;
	private final long totalMemory;
	private final long maxMemory;

	/**
	 * Instantiates a new MonitoringData object
	 *
	 * @param runGcBeforeCollecting
	 * 		boolean read from kernel properties to run GC before collecting monitoring data
	 */
	public MonitoringData(boolean runGcBeforeCollecting) {
		this.ramUsagePerModuleMap = new PackedMap<>();
		this.executionCounterPerModuleMap = new PackedMap<>();
		this.originalQuotaPerModuleMap = new PackedMap<>();
		// Init Memory Monitoring data
		this.runGcBeforeCollecting = runGcBeforeCollecting;
		if (runGcBeforeCollecting) {
			System.gc();// NOSONAR
		}
		this.totalMemory = Runtime.getRuntime().totalMemory();
		this.maxMemory = Runtime.getRuntime().maxMemory();
	}

	/**
	 * Gets the execution counter per module map
	 *
	 * @return the execution counter map
	 */
	public PackedMap<Module, ExecutionCounter> getExecutionCounterPerModuleMap() {
		return executionCounterPerModuleMap;
	}

	/**
	 * Gets the ram usage per module map
	 *
	 * @return the ram usage map
	 */
	public PackedMap<Module, Long> getRamUsagePerModuleMap() {
		return ramUsagePerModuleMap;
	}

	/**
	 * Gets the original quota per module map
	 *
	 * @return the original quota map
	 */
	public PackedMap<Module, Integer> getOriginalQuotaPerModuleMap() {
		return originalQuotaPerModuleMap;
	}

	/**
	 * Gets total memory value
	 *
	 * @return the total memory value
	 */
	public long getTotalMemory() {
		return totalMemory;
	}

	/**
	 * Gets the max memory value
	 *
	 * @return the max memory value
	 */
	public long getMaxMemory() {
		return maxMemory;
	}

	/**
	 * Gets the free memory value
	 *
	 * @return the free memory value
	 */
	public long getFreeMemory() {
		if (runGcBeforeCollecting) {
			System.gc(); // NOSONAR
		}
		return Runtime.getRuntime().freeMemory();
	}

	/**
	 * This class represents the execution counter
	 */
	public static class ExecutionCounter {
		private long current;
		private long last;

		public ExecutionCounter(long execCounter, long lastExecCounter) {
			this.current = execCounter;
			this.last = lastExecCounter;
		}

		public long getCurrent() {
			return current;
		}

		public void setCurrent(long current) {
			this.current = current;
		}

		public long getLast() {
			return last;
		}

		public void setLast(long last) {
			this.last = last;
		}
	}

}
