/*
 * Java
 *
 * Copyright 2021-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Tests blend modes.
 */
@SuppressWarnings("nls")
public class TestFontBlendMode {

	private static final float DEFAULT_LETTER_SPACING = 0f;
	private static final int ALPHA = 0x88;
	private static final String STRING_SQUARE = "A";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	private static Path createSquarePath() {
		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();
		return path;
	}

	/**
	 * Tests drawing a path with {@link BlendMode#SRC}.
	 */
	@Test
	public static void testFontBlendSrc() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		Path path = createSquarePath();
		String string = STRING_SQUARE;
		float fontSize = 100;
		int radius = 100;

		Matrix matrix = new Matrix();
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		matrix.setTranslate(51, 51);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, ALPHA, BlendMode.SRC,
				DEFAULT_LETTER_SPACING);

		TestUtilities.check("top left", 51, 51, g, 0x000088);
		TestUtilities.check("centerin", 99, 99, g, 0x000088);
		TestUtilities.check("centerout", 100, 100, g, 0x000088);
		TestUtilities.check("bottom right", 148, 148, g, 0x000088);

		int x = display.getWidth() / 4;
		int y = display.getHeight() / 2;

		matrix.reset();
		matrix.setTranslate(x + radius - 50, y - 50);
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		matrix.reset();
		matrix.setTranslate(x, y);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE, ALPHA,
				BlendMode.SRC, DEFAULT_LETTER_SPACING);
		display.flush();

		TestUtilities.check("centerin", x + radius + 49, y + 49, g, 0x000088);
		TestUtilities.check("centerout", x + radius + 51, y + 51, g, 0x000088);
	}

	/**
	 * Tests drawing a path with {@link BlendMode#SRC_OVER}.
	 */
	@Test
	public static void testFontBlendSourceOver() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE;
		float fontSize = 100;
		int radius = 100;
		Path path = createSquarePath();

		Matrix matrix = new Matrix();
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		matrix.setTranslate(51, 51);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.drawString(g, string, font, 100, matrix, ALPHA, BlendMode.SRC_OVER,
				DEFAULT_LETTER_SPACING);

		TestUtilities.check("top left", 51, 51, g, 0x700088);
		TestUtilities.check("centerin", 99, 99, g, 0x700088);
		TestUtilities.check("centerout", 100, 100, g, 0x000088);
		TestUtilities.check("bottom right", 148, 148, g, 0x000088);

		int x = display.getWidth() / 4;
		int y = display.getHeight() / 2;

		matrix.reset();
		matrix.setTranslate(x + radius - 50, y - 50);
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		matrix.reset();
		matrix.setTranslate(x, y);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE, ALPHA,
				BlendMode.SRC_OVER, DEFAULT_LETTER_SPACING);
		display.flush();

		TestUtilities.check("centerin", x + radius + 49, y + 49, g, 0x700088);
		TestUtilities.check("centerout", x + radius + 51, y + 51, g, 0x000088);
	}

	/**
	 * Tests drawing a path with {@link BlendMode#DST_OVER}.
	 */
	@Test
	public static void testFontBlendDstOver() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE;
		float fontSize = 100;
		int radius = 100;
		Path path = createSquarePath();

		Matrix matrix = new Matrix();
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		matrix.setTranslate(51, 51);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.drawString(g, string, font, 100, matrix, ALPHA, BlendMode.DST_OVER,
				DEFAULT_LETTER_SPACING);

		TestUtilities.check("top left", 51, 51, g, Colors.RED);
		TestUtilities.check("centerin", 99, 99, g, Colors.RED);
		TestUtilities.check("centerout", 100, 100, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("bottom right", 148, 148, g, TestUtilities.BACKGROUND_COLOR);

		int x = display.getWidth() / 4;
		int y = display.getHeight() / 2;

		matrix.reset();
		matrix.setTranslate(x + radius - 50, y - 50);
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		matrix.reset();
		matrix.setTranslate(x, y);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE, ALPHA,
				BlendMode.DST_OVER, DEFAULT_LETTER_SPACING);
		display.flush();

		TestUtilities.check("centerin", x + radius + 49, y + 49, g, Colors.RED);
		TestUtilities.check("centerout", x + radius + 51, y + 51, g, TestUtilities.BACKGROUND_COLOR);
	}

	/**
	 * Tests drawing a path with {@link BlendMode#SRC_IN}.
	 */
	@Test
	public static void testFontBlendSrcIn() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE;
		float fontSize = 100;
		int radius = 100;
		Path path = createSquarePath();

		Matrix matrix = new Matrix();
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		matrix.setTranslate(51, 51);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.drawString(g, string, font, 100, matrix, ALPHA, BlendMode.SRC_IN, DEFAULT_LETTER_SPACING);

		TestUtilities.check("top left", 51, 51, g, 0x000088);
		TestUtilities.check("centerin", 99, 99, g, 0x000088);
		TestUtilities.check("centerout", 100, 100, g, 0x000088);
		TestUtilities.check("bottom right", 148, 148, g, 0x000088);

		int x = display.getWidth() / 4;
		int y = display.getHeight() / 2;

		matrix.reset();
		matrix.setTranslate(x + radius - 50, y - 50);
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		matrix.reset();
		matrix.setTranslate(x, y);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE, ALPHA,
				BlendMode.SRC_IN, DEFAULT_LETTER_SPACING);
		display.flush();

		TestUtilities.check("centerin", x + radius + 49, y + 49, g, 0x000088);
		TestUtilities.check("centerout", x + radius + 51, y + 51, g, 0x000088);
	}

	// /**
	// * Tests drawing a path with {@link BlendMode#DST_IN}.
	// */
	// @Test
	// public static void testFontBlendDstIn() {
	// Display display = Display.getDisplay();
	// GraphicsContext g = display.getGraphicsContext();
	// VectorFont font = TestUtilities.getTestFont();
	// String string = STRING_SQUARE;
	// float fontSize = 100;
	// int radius = 100;
	// Path path = createSquarePath();
	//
	// Matrix matrix = new Matrix();
	// g.setColor(Colors.RED);
	// VectorGraphicsPainter.fillPath(g, path, matrix);
	//
	// g.setColor(Colors.BLUE);
	// VectorGraphicsPainter.drawString(g, string, font, 100, 51, 51, ALPHA, BlendMode.DST_IN, DEFAULT_LETTER_SPACING);
	//
	// TestUtilities.check("top left", 51, 51, g, 0x880000);
	// TestUtilities.check("centerin", 99, 99, g, 0x880000);
	// TestUtilities.check("centerout", 100, 100, g, TestUtilities.BACKGROUND_COLOR);
	// TestUtilities.check("bottom right", 148, 148, g, TestUtilities.BACKGROUND_COLOR);
	//
	// int x = display.getWidth() / 4;
	// int y = display.getHeight() / 2;
	//
	// matrix.reset();
	// matrix.setTranslate(x + radius - 50, y - 50);
	// g.setColor(Colors.RED);
	// VectorGraphicsPainter.fillPath(g, path, matrix);
	//
	// matrix.reset();
	// matrix.setTranslate(x, y);
	// g.setColor(Colors.BLUE);
	// VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
	// ALPHA, BlendMode.DST_IN, 0);
	// display.flush();
	//
	// TestUtilities.check("centerin", x + radius + 49, y + 49, g, 0x880000);
	// TestUtilities.check("centerout", x + radius + 51, y + 51, g, TestUtilities.BACKGROUND_COLOR);
	// }
	//
	// /**
	// * Tests drawing a path with {@link BlendMode#SCREEN}.
	// */
	// @Test
	// public static void testFontBlendScreen() {
	// Display display = Display.getDisplay();
	// GraphicsContext g = display.getGraphicsContext();
	// VectorFont font = TestUtilities.getTestFont();
	// String string = STRING_SQUARE;
	// float fontSize = 100;
	// int radius = 100;
	// Path path = createSquarePath();
	//
	// Matrix matrix = new Matrix();
	// g.setColor(Colors.RED);
	// VectorGraphicsPainter.fillPath(g, path, matrix);
	//
	// g.setColor(Colors.BLUE);
	// VectorGraphicsPainter.drawString(g, string, font, 100, 51, 51, ALPHA, BlendMode.SCREEN, DEFAULT_LETTER_SPACING);
	//
	// TestUtilities.check("top left", 51, 51, g, Colors.MAGENTA);
	// TestUtilities.check("centerin", 99, 99, g, Colors.MAGENTA);
	// TestUtilities.check("centerout", 100, 100, g, Colors.BLUE);
	// TestUtilities.check("bottom right", 148, 148, g, Colors.BLUE);
	//
	// int x = display.getWidth() / 4;
	// int y = display.getHeight() / 2;
	//
	// matrix.reset();
	// matrix.setTranslate(x + radius - 50, y - 50);
	// g.setColor(Colors.RED);
	// VectorGraphicsPainter.fillPath(g, path, matrix);
	//
	// matrix.reset();
	// matrix.setTranslate(x, y);
	// g.setColor(Colors.BLUE);
	// VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
	// ALPHA, BlendMode.SCREEN, 0);
	// display.flush();
	//
	// TestUtilities.check("centerin", x + radius + 49, y + 49, g, Colors.MAGENTA);
	// TestUtilities.check("centerout", x + radius + 51, y + 51, g, Colors.BLUE);
	// }
	//
	// /**
	// * Tests drawing a path with {@link BlendMode#MULTIPLY}.
	// */
	// @Test
	// public static void testFontBlendMultiply() {
	// Display display = Display.getDisplay();
	// GraphicsContext g = display.getGraphicsContext();
	// VectorFont font = TestUtilities.getTestFont();
	// String string = STRING_SQUARE;
	// float fontSize = 100;
	// int radius = 100;
	// Path path = createSquarePath();
	//
	// Matrix matrix = new Matrix();
	// g.setColor(Colors.RED);
	// VectorGraphicsPainter.fillPath(g, path, matrix);
	//
	// g.setColor(Colors.BLUE);
	// VectorGraphicsPainter.drawString(g, string, font, 100, 51, 51, ALPHA, BlendMode.MULTIPLY,
	// DEFAULT_LETTER_SPACING);
	//
	// TestUtilities.check("top left", 51, 51, g, 0x770000);
	// TestUtilities.check("centerin", 99, 99, g, 0x770000);
	// TestUtilities.check("centerout", 100, 100, g, TestUtilities.BACKGROUND_COLOR);
	// TestUtilities.check("bottom right", 148, 148, g, TestUtilities.BACKGROUND_COLOR);
	//
	// int x = display.getWidth() / 4;
	// int y = display.getHeight() / 2;
	//
	// matrix.reset();
	// matrix.setTranslate(x + radius - 50, y - 50);
	// g.setColor(Colors.RED);
	// VectorGraphicsPainter.fillPath(g, path, matrix);
	//
	// matrix.reset();
	// matrix.setTranslate(x, y);
	// g.setColor(Colors.BLUE);
	// VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
	// ALPHA, BlendMode.MULTIPLY, 0);
	// display.flush();
	//
	// TestUtilities.check("centerin", x + radius + 49, y + 49, g, 0x770000);
	// TestUtilities.check("centerout", x + radius + 51, y + 51, g, TestUtilities.BACKGROUND_COLOR);
	// }

}
