/*
 * Java
 *
 * Copyright 2021-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorImage;

/**
 * Tests the drawing of images.
 */
@SuppressWarnings("nls")
public class TestDrawImage {

	private static final String MASCOT = "/com/microej/microvg/test/mascot.xml";
	private static final String RED_RECTANGLE = "/com/microej/microvg/test/red_rectangle.xml";
	private static final String GRADIENT_RECTANGLE = "/com/microej/microvg/test/gradient.xml";
	private static final String GROUP = "/com/microej/microvg/test/group.xml";
	private static final String CLIP = "/com/microej/microvg/test/clip.xml";
	private static final String GRADIENTS = "/com/microej/microvg/test/multiple_gradients.xml";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
		Display.getDisplay().getGraphicsContext().reset();
	}

	/**
	 * Tests drawing an image.
	 */
	@Test
	public static void testDrawImage() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(MASCOT);
		VectorGraphicsPainter.drawImage(g, image, new Matrix());

		display.flush();

		Assert.assertEquals("mascot width", 400, image.getWidth(), 0f);
		Assert.assertEquals("mascot height", 389f, image.getHeight(), 0f);

		TestUtilities.check("top mascot", 212, 101, g, 0xee502e);
		TestUtilities.check("center mascot", 212, 200, g, Colors.WHITE);
		TestUtilities.check("bottom mascot", 212, 300, g, 0xcf4520);
		TestUtilities.check("left mascot", 110, 240, g, 0xee502e);
		TestUtilities.check("right mascot", 310, 240, g, 0xee502e);
	}

	/**
	 * Tests drawing an image with x,y parameters.
	 */
	@Test
	public static void testDrawImageXY() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(MASCOT);
		VectorGraphicsPainter.drawImage(g, image, 20, 40);

		display.flush();

		Assert.assertEquals("mascot width", 400, image.getWidth(), 0f);
		Assert.assertEquals("mascot height", 389f, image.getHeight(), 0f);

		TestUtilities.check("top mascot", 232, 141, g, 0xee502e);
		TestUtilities.check("center mascot", 232, 240, g, Colors.WHITE);
		TestUtilities.check("bottom mascot", 232, 340, g, 0xcf4520);
		TestUtilities.check("left mascot", 130, 280, g, 0xee502e);
		TestUtilities.check("right mascot", 330, 280, g, 0xee502e);
	}

	/**
	 * Tests drawing an image with a matrix.
	 */
	@Test
	public static void testDrawImageMatrix() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(MASCOT);
		Matrix matrix = new Matrix();
		matrix.setTranslate(80, -50);
		matrix.preScale(.8f, .8f);
		matrix.preRotate(45);
		VectorGraphicsPainter.drawImage(g, image, matrix);

		display.flush();

		TestUtilities.check("top mascot", 143, 142, g, 0xee502e);
		TestUtilities.check("center mascot", 90, 180, g, Colors.WHITE);
		TestUtilities.check("bottom mascot", 25, 245, g, 0xcf4520);
		TestUtilities.check("left mascot", 5, 145, g, 0xee502e);
		TestUtilities.check("right mascot", 120, 265, g, 0xee502e);
	}

	/**
	 * Tests drawing an image with transparency.
	 */
	@Test
	public static void testDrawImageTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());

		VectorImage image = VectorImage.getImage(RED_RECTANGLE);
		VectorGraphicsPainter.drawImage(g, image, new Matrix(), 0x88);

		display.flush();

		TestUtilities.check("top left", 0, 0, g, Colors.BLUE);
		TestUtilities.check("intersection", 75, 75, g, 0x880077);
		TestUtilities.check("bottom right", 149, 149, g, 0x880000);
	}

	/**
	 * Tests drawing an image with overlapping paths with transparency.
	 */
	@Test
	public static void testDrawImageTransparentOverlap() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(MASCOT);
		boolean exception;
		try {
			VectorGraphicsPainter.drawImage(g, image, new Matrix(), 0x88);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);
	}

	/**
	 * Tests drawing an image with gradient paths and matrix
	 */
	@Test
	public static void testDrawImageGradient() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE);
		VectorGraphicsPainter.drawImage(g, image, new Matrix());

		display.flush();

		Assert.assertEquals("gradient width", 101f, image.getWidth(), 0f);
		Assert.assertEquals("gradient height", 101f, image.getHeight(), 0f);

		TestUtilities.check("top", 50, 0, g, Colors.RED);
		TestUtilities.check("center", 50, 50, g, Colors.LIME);
		TestUtilities.check("bottom", 50, 99, g, Colors.BLUE);
	}

	/**
	 * Tests drawing an image with gradient paths.
	 */
	@Test
	public static void testDrawImageGradientMatrix() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE);

		Matrix matrix = new Matrix();
		// 90° rotation over center of cube
		matrix.setTranslate(-50, -50);
		matrix.postRotate(90);
		matrix.postTranslate(50, 50);
		// scale x2
		matrix.postScale(2f, 2f);
		// translate to [100,100]
		matrix.postTranslate(100, 100);
		VectorGraphicsPainter.drawImage(g, image, matrix);

		display.flush();

		Assert.assertEquals("gradient width", 101f, image.getWidth(), 0f);
		Assert.assertEquals("gradient height", 101f, image.getHeight(), 0f);

		TestUtilities.check("right", 299, 200, g, Colors.RED);
		TestUtilities.check("center", 200, 200, g, Colors.LIME);
		TestUtilities.check("left", 101, 200, g, Colors.BLUE);
	}

	/**
	 * Tests drawing an image containing a gradient with transparency.
	 */
	@Test
	public static void testDrawImageGradientTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.lineTo(150, 50);
		path.lineTo(150, 150);
		path.lineTo(50, 150);
		g.setColor(Colors.RED);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE);
		VectorGraphicsPainter.drawImage(g, image, new Matrix(), 0x88);

		display.flush();

		TestUtilities.check("top left", 0, 0, g, 0x880000);
		TestUtilities.check("center left", 0, 50, g, 0x008800);
		TestUtilities.check("bottom left", 0, 99, g, 0x000088);
		TestUtilities.check("center intersection", 75, 50, g, 0x788800);
		TestUtilities.check("bottom intersection", 75, 99, g, 0x780088);
	}

	/**
	 * Tests drawing an image that contains a group.
	 */
	@Test
	public static void testDrawImageGroup() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GROUP);
		VectorGraphicsPainter.drawImage(g, image, new Matrix());

		display.flush();

		TestUtilities.check("top left first", 10, 10, g, Colors.BLUE);
		TestUtilities.check("bottom right first", 109, 109, g, Colors.MAGENTA);
		TestUtilities.check("top left second", 60, 120, g, 0x7f0000);
		TestUtilities.check("bottom right second", 159, 169, g, 0x7f0000);
		TestUtilities.check("top left third", 100, 185, g, Colors.LIME);
		TestUtilities.check("bottom right third", 199, 284, g, Colors.LIME);
	}

	/**
	 * Tests drawing an image that contains a group with transparency.
	 */
	@Test
	public static void testDrawImageGroupTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(80, 80);
		path.lineTo(130, 80);
		path.lineTo(130, 220);
		path.lineTo(80, 220);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.fillPath(g, path, new Matrix());

		VectorImage image = VectorImage.getImage(GROUP);
		VectorGraphicsPainter.drawImage(g, image, new Matrix(), 0x88);

		display.flush();

		TestUtilities.check("top left first", 10, 10, g, 0x000088);
		TestUtilities.check("top right first", 109, 10, g, 0x880088);
		TestUtilities.check("bottom right first", 109, 109, g, 0x8800ff);
		TestUtilities.check("top left second", 60, 120, g, 0x440000);
		TestUtilities.check("center second", 110, 145, g, 0x4400bb);
		TestUtilities.check("bottom right second", 159, 169, g, 0x440000);
		TestUtilities.check("top left third", 100, 185, g, 0x008878);
		TestUtilities.check("bottom left third", 100, 284, g, 0x008800);
		TestUtilities.check("bottom right third", 199, 284, g, 0x008800);
	}

	/**
	 * Tests drawing an image with image bounding box smaller than image content.
	 */
	@Test
	public static void testDrawImageClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(CLIP);

		Matrix matrix = new Matrix();

		int x = 100;
		int y = 100;
		int width = 100;
		int height = 100;
		int testThickness = 1;
		int errors = 0;

		matrix.postTranslate(100, 100);
		VectorGraphicsPainter.drawImage(g, image, matrix);

		display.flush();
		errors += TestUtilities.checkPeripheralArea("clip fit inside", Colors.BLUE, x + testThickness,
				y + testThickness, width - 2 * testThickness, height - 2 * testThickness, testThickness, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip fit outside", TestUtilities.BACKGROUND_COLOR, x, y, width,
				height, testThickness, 0, false);

		assertEquals(0, errors);
	}

	/**
	 * Tests drawing an image with a translation and a clip
	 */
	@Test
	public static void testDrawImageClipAndTranslation() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(CLIP);

		Matrix matrix = new Matrix();

		int x = 100;
		int y = 100;
		int width = 50;
		int height = 50;
		int testThickness = 1;
		int errors = 0;

		g.translate(x, y);
		g.setClip(0, 0, width, height);
		VectorGraphicsPainter.drawImage(g, image, matrix);
		g.reset();

		display.flush();
		errors += TestUtilities.checkPeripheralArea("clip fit inside", Colors.BLUE, x + testThickness,
				y + testThickness, width - 2 * testThickness, height - 2 * testThickness, testThickness, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip fit outside", TestUtilities.BACKGROUND_COLOR, x, y, width,
				height, testThickness, 0, false);

		assertEquals(0, errors);
	}

	/**
	 * Tests drawing an image with several consecutive gradients: the same gradient must not be used to draw all
	 * gradients
	 */
	@Test
	public static void testDrawImageSeveralGradients() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.translate(50, 50);

		VectorImage image = VectorImage.getImage(GRADIENTS);

		Matrix matrix = new Matrix();

		matrix.postScale(10, 10);
		VectorGraphicsPainter.drawImage(g, image, matrix);

		display.flush();

		int background = g.readPixel(12, 80);

		int gradient1left = g.readPixel(21, 80);
		int gradient2left = g.readPixel(50, 80);
		int gradient3left = g.readPixel(79, 80);
		int gradient4left = g.readPixel(108, 80);

		int gradient1right = g.readPixel(41, 80);
		int gradient2right = g.readPixel(70, 80);
		int gradient3right = g.readPixel(99, 80);
		int gradient4right = g.readPixel(128, 80);

		Assert.assertFalse("gradient 1 should be drawn (left)", TestUtilities.compare(background, gradient1left));
		Assert.assertFalse("gradient 2 should be drawn (left)", TestUtilities.compare(background, gradient2left));
		Assert.assertFalse("gradient 3 should be drawn (left)", TestUtilities.compare(background, gradient3left));
		Assert.assertFalse("gradient 4 should be drawn (left)", TestUtilities.compare(background, gradient4left));

		Assert.assertFalse("gradient 1 should be drawn (right)", TestUtilities.compare(background, gradient1right));
		Assert.assertFalse("gradient 2 should be drawn (right)", TestUtilities.compare(background, gradient2right));
		Assert.assertFalse("gradient 3 should be drawn (right)", TestUtilities.compare(background, gradient3right));
		Assert.assertFalse("gradient 4 should be drawn (right)", TestUtilities.compare(background, gradient4right));

		Assert.assertFalse("gradient 1 left and right colors should be different",
				TestUtilities.compare(gradient1left, gradient1right));
		Assert.assertFalse("gradient 2 left and right colors should be different",
				TestUtilities.compare(gradient2left, gradient2right));
		Assert.assertFalse("gradient 3 left and right colors should be different",
				TestUtilities.compare(gradient3left, gradient3right));
		Assert.assertFalse("gradient 4 left and right colors should be different",
				TestUtilities.compare(gradient4left, gradient4right));

		Assert.assertFalse("gradient 1 and 2 should be different",
				TestUtilities.compare(gradient1right, gradient2right));
		Assert.assertFalse("gradient 2 and 3 should be different",
				TestUtilities.compare(gradient2right, gradient3right));
		Assert.assertFalse("gradient 3 and 4 should be different",
				TestUtilities.compare(gradient3right, gradient4right));

	}

	/**
	 * Tests drawing an image after a flush.
	 */
	@Test
	public static void testDrawImageAndFlush() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		int w = g.getWidth();
		int h = g.getHeight();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE);

		// ensure to erase all buffers
		g.setColor(Colors.BLACK);
		for (int i = 6 /* multiple of double and triple buffers */; --i >= 0;) {
			Painter.fillRectangle(g, 0, 0, w, h);
			display.flush();
		}

		// draw image: the destination must be configured again (not the same destination as for the previous fillrect)
		VectorGraphicsPainter.drawImage(g, image, 0, 0);
		display.flush();

		// draw with restoring
		g.setTranslation(w / 2, 0);
		g.setClip(0, 0, w / 2, h);
		VectorGraphicsPainter.drawImage(g, image, 0, 0);

		display.flush();

		// the first image must be restored
		g.reset();
		TestUtilities.check("top left", 50, 50, g, Colors.RED);
		TestUtilities.check("bottom right", 149, 149, g, Colors.RED);

	}

}
