/*
 * Java
 *
 * Copyright 2019-2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.concurrent.atomic.AtomicInteger;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.Widget;
import ej.fp.Widget.WidgetAttribute;

/**
 * This interface holds all methods required by MicroUI implementation to be compatible with the MicroUI Graphics
 * Engine.
 * <p>
 * An implementation of this interface is required to run a MicroUI application which uses the class MicroUI Display;
 * otherwise some errors will be thrown at runtime and the application will not able to run correctly. This
 * implementation should be a widget which allows to draw in an {@link Image}. This image is used as "back buffer" (see
 * {@link #isDoubleBuffered()}). At the end of MicroEJ application drawings (via MicroUI painter classes), a call to
 * {@link #flush()} is performed to render the image on the visible part of front panel.
 * <p>
 * The display size may be different than the widget size. This allows to show on computer a bigger or smaller display
 * in pixels than the embedded display. The display size is the size given to the MicroEJ application. To retrieve the
 * display size, the methods {@link Image#getWidth()} and {@link Image#getHeight()} (on the image returned by
 * {@link #initialize()}) are called (instead of {@link Widget#getWidth()} and {@link Widget#getHeight()} which return
 * the widget size). The both values can be automatically set by the front panel parser if the display widget
 * (implementation of this interface) declares the following {@link WidgetAttribute}:
 *
 * <pre>
 * &#64;WidgetAttribute(name = "displayWidth", isOptional = true)
 * &#64;WidgetAttribute(name = "displayHeight", isOptional = true)
 * </pre>
 *
 * This size may be optional, in this case the display size is equal to the widget size. This size (if applicable the
 * widget size) must be used to create the {@link Image} returned by {@link #initialize()}.
 * <p>
 * This interface stubs a lot of methods. Only {@link #initialize()} and {@link #flush()} are mandatory. All others
 * methods are optional and perform same defaut behavior like embedded side.
 */
public interface LLUIDisplayImpl {

	/**
	 * Minimal value of backlight according MicroUI specification.
	 */
	int BACKLIGHT_MIN = 0;

	/**
	 * Maximal value of backlight according MicroUI specification.
	 */
	int BACKLIGHT_MAX = 100;

	/**
	 * Minimal value of contrast according MicroUI specification.
	 */
	int CONTRAST_MIN = 0;

	/**
	 * Maximal value of contrast according MicroUI specification.
	 */
	int CONTRAST_MAX = 100;

	/**
	 * Initializes the display widget to be compatible with MicroUI Graphics Engine. Called by framework when MicroEJ
	 * application is calling <code>MicroUI.start()</code>.
	 * <p>
	 * This method has to return an image where MicroUI will performing its drawings. As mentioned in class comment, the
	 * widget size can be higher or smaller than the simulated display (embedded display). The returned buffer must have
	 * the embedded display size (and not the widget size).
	 * <p>
	 * This buffer is given as parameter in {@link #flush()} method.
	 *
	 * @return the buffer where performing the MicroUI drawing.
	 */
	Image initialize();

	/**
	 * Flushes a part of the display limited by the specified bounds when the display is double buffered.
	 *
	 * This display's flush may be synchronous (the implementation updates the front buffer immediately) or asynchronous
	 * (the implementation uses a thread to simulate the embedded side). In second case, the call to
	 * {@link #waitFlush()} must synchronize the Graphics Engine with the widget display.
	 *
	 * If the display does not have a <code>backBuffer</code> (not double buffered), nothing has to be done.
	 */
	void flush();

	/**
	 * Notifies that a region will be modified by the application in the display back buffer.
	 * <p>
	 * This function is called by the Graphics Engine:
	 * <ul>
	 * <li>For each call to {@link MicroUIGraphicsContext#requestDrawing()} (in the display back buffer) if the drawing
	 * region set by the application is different than the previous drawing region. In this case, the
	 * <code>drawingNow</code> parameter is set to <code>true</code>.</li>
	 * <li>For each call to GraphicsContext.notifyDirtyRegion(). In this case, the <code>drawingNow</code> parameter is
	 * set to <code>false</code>.</li>
	 * </ul>
	 *
	 * @param x1
	 *            the left x coordinate of the drawing region.
	 * @param y1
	 *            the top y coordinate of the drawing region.
	 * @param x2
	 *            the right x coordinate of the drawing region.
	 * @param y2
	 *            the bottom y coordinate of the drawing region.
	 * @param drawingNow
	 *            <code>true</code> if a drawing is following this call, <code>false</code> otherwise.
	 */
	void newDrawingRegion(int x1, int y1, int x2, int y2, boolean drawingNow);

	/**
	 * Simulates the waiting of end of an asynchronous flush when the display is double buffered.
	 *
	 * When the call to {@link #flush()} performs a synchronous flush, this method should stay empty because there is
	 * nothing to wait (default implementation).
	 *
	 * When the call to {@link #flush()} performs an asynchronous flush (to simulate the same behavior than embedded
	 * side): this method is useful to wait the end of this asynchronous flush.
	 *
	 * If the display does not have a <code>backBuffer</code> (not double buffered), nothing is done. This method should
	 * stay empty.
	 *
	 * @param gc
	 *            the flushed graphics context.
	 */
	default void waitFlush() {
		// default implementation: flush is not asynchronous: there is nothing to wait
	}

	/**
	 * Gets the current back buffer (the buffer when the Graphics Engine has to draw).
	 *
	 * @return the current back buffer.
	 */
	default Image getCurrentBackBuffer() {
		// call deprecated method for backward compatibility with UI Pack 14.0.0
		return getCurrentDrawingBuffer();
	}

	/**
	 * @deprecated Implements {@link #getCurrentBackBuffer()} instead.
	 *
	 * @return the current back buffer.
	 */
	@SuppressWarnings("nls")
	@Deprecated
	default Image getCurrentDrawingBuffer() {
		throw new RuntimeException("LLUIDisplayImpl.getCurrentBackBuffer() must be implemented");
	}

	/**
	 * Gets the image currently displayed (with or without debug tools, backlight, contrast, etc.).
	 *
	 * @return an image that represents the content of the display currently visible
	 */
	Image getDisplayedImage();

	/**
	 * Sets the contrast of the display. By default this method does nothing (feature not supported).
	 *
	 * @param contrast
	 *            the new value of the contrast (range {@link #CONTRAST_MIN}-{@link #CONTRAST_MAX}).
	 */
	default void setContrast(int contrast) {
		// feature not supported
	}

	/**
	 * Returns the contrast of the display. By default this method returns {@link #CONTRAST_MIN}.
	 *
	 * @return the current contrast of the display (range {@link #CONTRAST_MIN}-{@link #CONTRAST_MAX}).
	 */
	default int getContrast() {
		return CONTRAST_MIN;
	}

	/**
	 * Returns true if the display holds a backlight. By default this method returns false (feature not supported).
	 *
	 * @return true when widget manages display backlight.
	 */
	default boolean hasBacklight() {
		return false;
	}

	/**
	 * Sets the backlight of the display. By default this method does nothing (feature not supported).
	 *
	 * @param backlight
	 *            the new value of the backlight (range {@link #BACKLIGHT_MIN}-{@link #BACKLIGHT_MAX}).
	 */
	default void setBacklight(int backlight) {
		// feature not supported
	}

	/**
	 * Returns the backlight of the display. By default this method returns {@link #BACKLIGHT_MIN}.
	 *
	 * @return the backlight of the display (range {@link #BACKLIGHT_MIN}-{@link #BACKLIGHT_MAX}).
	 */
	default int getBacklight() {
		return BACKLIGHT_MIN;
	}

	/**
	 * Asks if the display is a colored display or not.
	 * <p>
	 * By default this method returns false when the display pixel depth is lower than or equals to 4 and true
	 * otherwise.
	 *
	 * @return true when the display is not a grayscale display, false otherwise.
	 */
	default boolean isColor() {
		return LLUIDisplay.Instance.getDisplayPixelDepth() > 4;
	}

	/**
	 * Gets the number of colors that can be represented on the device.
	 * <p>
	 * Note that the number of colors for a black and white display is 2. Usually the number of colors is 1 << BPP (BPP
	 * without transparency bits).
	 * <p>
	 * By default this method returns a value which follows this default rule.
	 *
	 * @return the number of colors of the display.
	 */
	default int getNumberOfColors() {
		return 1 << LLUIDisplay.Instance.getDisplayPixelDepth();
	}

	/**
	 * Returns if the display uses an underlying double buffer.
	 * <p>
	 * By default this method returns true and considers the interface implementation returns a "back" buffer in
	 * {@link #initialize()}. This buffer is used to render the MicroUI drawings in background. When MicroUI
	 * Display.flush() method is called, a call to {@link #flush()} is performed. The implementation has to copy the
	 * image content to another image (visible on front panel).
	 * <p>
	 * Implementation can override this method to return false. In this case the image returned by {@link #initialize()}
	 * is used as rendering buffer for MicroUI drawings and as visible buffer on front panel at same time.
	 *
	 * @return true if the display is double buffered, else otherwise.
	 */
	default boolean isDoubleBuffered() {
		return true;
	}

	/**
	 * Converts the 32-bit ARGB color format (A-R-G-B) into the display color format.
	 * <p>
	 * This function is called only when the display is not a standard display: when the pixel data does not match with
	 * one of these formats: {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB8888} ,
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB888}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB565},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB1555},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB4444}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C4},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C2} or {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C1}.
	 * <p>
	 * In case of this function is not implemented whereas it is required, the result of pixel drawing is unknown.
	 * <p>
	 * Note: the opacity level (alpha) may be ignored if the display pixel representation does not hold the opacity
	 * level information.
	 * <p>
	 * The implementation should not directly call this function when performing a drawing. It must call
	 * {@link LLUIDisplay#convertARGBColorToColorToDraw(int)} instead in case of this conversion is Graphics Engine
	 * built-in (standard display).
	 *
	 * @param argbColor
	 *            the color to convert.
	 * @return the display color equivalent to <code>microUIColor</code>.
	 */
	default int convertARGBColorToDisplayColor(int argbColor) {
		return argbColor;
	}

	/**
	 * Converts the display color format into a 32-bit ARGB color format (A-R-G-B).
	 * <p>
	 * This function is called only when the display is not a standard display: when the pixel data does not match with
	 * one of these formats: {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB8888} ,
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB888}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB565},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB1555},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB4444}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C4},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C2} or {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C1}.
	 * <p>
	 * In case of this function is not implemented whereas it is required, the result of pixel drawing is unknown.
	 * <p>
	 * Note: the opacity level (alpha) may be ignored if the display pixel representation does not hold the opacity
	 * level information. In this case, the returned alpha level is 0xff (full opaque).
	 *
	 * @param displayColor
	 *            the color to convert.
	 * @return the MicroUI color equivalent to <code>displayColor</code>.
	 */
	default int convertDisplayColorToARGBColor(int displayColor) {
		return displayColor;
	}

	/**
	 * Prepares the blending of two ARGB colors.
	 * <p>
	 * This function is called only when the display is not a standard display: when the pixel data does not match with
	 * one of these formats: {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB8888} ,
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB888}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_RGB565},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB1555},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_ARGB4444}, {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C4},
	 * {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C2} or {@link MicroUIImageFormat#MICROUI_IMAGE_FORMAT_C1}.
	 * <p>
	 * This is useful only when the LCD is a palletized LCD. This method is called by framework when the MicroEJ
	 * application draws something which requires a blending between the current foreground color and a specific
	 * background color (draw a string, draw an anti-aliased line etc.).
	 * <p>
	 * The implementation has to replace the ARGB colors by the indexes of these colors in the LCD CLUT. The framework
	 * will use the intermediate values between these two indexes instead of blending in software the ARGB colors. No
	 * post conversion will be performed later.
	 * <p>
	 * When the ARGB colors are not available in the CLUT or when the range specified by the two ARGB colors is not
	 * expected by the CLUT, the implementation should return false (feature not supported). In this way the blending
	 * will be done in software and the result color will be converted later thanks a call to
	 * {@link #convertARGBColorToDisplayColor(int)}.
	 * <p>
	 * By default the method returns false (feature not supported / useless).
	 *
	 * @param foreground
	 *            the foreground ARGB color to convert.
	 * @param background
	 *            the background ARGB color to convert.
	 * @return true when the indexes have been found, false otherwise.
	 */
	default boolean prepareBlendingOfIndexedColors(AtomicInteger foreground, AtomicInteger background) {
		return false;
	}

	/**
	 * Decodes the image defined by the <code>data</code> array.
	 * <p>
	 * The default implementation tries to decode the image by using the platform-specific loader.
	 *
	 * @param data
	 *            the data of the image to decode.
	 * @return the device-specific {@link Image} created.
	 * @throws IOException
	 *             if an error occur during the decoding of the image.
	 */
	default Image decode(byte[] data) throws IOException {
		return FrontPanel.getFrontPanel().newImage(new ByteArrayInputStream(data));
	}

}
