/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.microui.display.GraphicsContext;
import ej.microui.util.EventHandler;

/**
 * A renderable is an object that is intended to be rendered on the screen.
 */
public interface Renderable {

	/**
	 * Gets the x coordinate of this renderable, relative to its parent.
	 *
	 * @return the x coordinate
	 */
	int getX();

	/**
	 * Gets the y coordinate of this renderable, relative to its parent.
	 *
	 * @return the y coordinate
	 */
	int getY();

	/**
	 * Gets the width of this renderable.
	 *
	 * @return the width
	 */
	int getWidth();

	/**
	 * Gets the height of this renderable.
	 *
	 * @return the height
	 */
	int getHeight();

	/**
	 * Gets whether or not this renderable is shown on a display.
	 *
	 * @return <code>true</code> if the renderable is shown, <code>false</code> otherwise
	 */
	boolean isShown();

	/**
	 * This method is called as soon as the renderable is visible.
	 */
	void showNotify();

	/**
	 * This method is called as soon as the renderable is hidden.
	 */
	void hideNotify();

	/**
	 * Requests a repaint of this renderable.
	 */
	void repaint();

	/**
	 * Requests a repaint of a zone of this renderable.
	 *
	 * @param x
	 *            the relative x coordinate of the area to repaint
	 * @param y
	 *            the relative y coordinate of the area to repaint
	 * @param width
	 *            the width of the area to repaint
	 * @param height
	 *            the height of the area to repaint
	 */
	void repaint(int x, int y, int width, int height);

	/**
	 * Renders the renderable on the given graphics context.
	 *
	 * @param g
	 *            the graphics context to be used to draw the renderable.
	 * @since 2.0
	 */
	void render(@NonNull GraphicsContext g);

	/**
	 * Sets the event handler.
	 *
	 * @param eventHandler
	 *            the event handler to set.
	 * @see #handleEvent(int)
	 * @since 2.0
	 */
	void setEventHandler(@Nullable EventHandler eventHandler);

	/**
	 * Gets the event handler.
	 *
	 * @return the event handler.
	 * @see #handleEvent(int)
	 * @since 2.0
	 */
	@Nullable
	EventHandler getEventHandler();

	/**
	 * Called by the system when an event occurred.<br>
	 * Pointer related events are sent to the pointed renderable. Other events are sent to the focus owner of the active
	 * panel and progressing up its hierarchy while not consumed until the desktop.
	 *
	 * @param event
	 *            the event to handle
	 * @return <code>true</code> if the renderable has consumed the event, <code>false</code> otherwise
	 * @since 0.9
	 */
	boolean handleEvent(int event);

}
