/*
 * Copyright 2018-2019 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.property.registry;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.property.PropertyPermission;
import ej.property.PropertyRegistry;

/**
 * Simple implementation of a property registry.
 */
public class SimplePropertyRegistry implements PropertyRegistry {

	private final PackedMap<String, String> properties;

	/**
	 * Creates a simple property registry.
	 */
	public SimplePropertyRegistry() {
		this.properties = new PackedMap<>();
	}

	@Override
	@Nullable
	public String getProperty(String key) {
		checkPropertyPermission(key, PropertyPermission.GET_ACTION);
		return this.properties.get(key);
	}

	@Override
	public String getProperty(String key, String defaultValue) {
		checkPropertyPermission(key, PropertyPermission.GET_ACTION);
		String value = this.properties.get(key);
		if (value != null) {
			return value;
		}
		return defaultValue;
	}

	@Override
	@Nullable
	public String setProperty(String key, String value) {
		checkPropertyPermission(key, PropertyPermission.SET_ACTION);
		String previous = getProperty(key);
		this.properties.put(key, value);
		return previous;
	}

	@Override
	@Nullable
	public String removeProperty(String key) {
		checkPropertyPermission(key, PropertyPermission.REMOVE_ACTION);
		return this.properties.remove(key);
	}

	/**
	 * Checks that an action on a property is permitted.
	 *
	 * @param key
	 *            the property key.
	 * @param action
	 *            the action.
	 */
	protected void checkPropertyPermission(String key, String action) {
		SecurityManager securityManager = System.getSecurityManager();
		if (securityManager != null) {
			@SuppressWarnings("null") // Class name is not null for sure.
			@NonNull
			String className = getClass().getSimpleName();
			securityManager.checkPermission(new PropertyPermission(className, key, action));
		}
	}

}
