/*
 * Copyright 2019-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.storage.fs;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import ej.annotation.Nullable;
import ej.storage.Storage;
import ej.storage.util.StorageHelper;

/**
 * Implementation of persistent storage on file system.
 */
public class StorageFs implements Storage {

	private static final String ROOT_PROPERTY_EXTENSION = ".root"; //$NON-NLS-1$
	private static final String DEFAULT_ROOT_FOLDER = "storage"; //$NON-NLS-1$

	private final File root;

	/**
	 * Creates a storage on file system.
	 * <p>
	 * The root folder where the files will be created can be specified with the property
	 * <code>ej.storage.fs.StorageFs.root</code>. If not set, <code>{@value #DEFAULT_ROOT_FOLDER}</code> will be used.
	 *
	 * @throws IOException
	 *             if the specified root already exists and is not a directory.
	 * @throws IOException
	 *             if the specified root cannot be created.
	 */
	public StorageFs() throws IOException {
		this(getRootFolderPropertyOrDefault());
	}

	/**
	 * Creates a storage on file system specifying the root folder where the files will be created.
	 *
	 * @param root
	 *            the root.
	 * @throws IOException
	 *             if the specified root already exists and is not a directory.
	 * @throws IOException
	 *             if the specified root cannot be created.
	 */
	public StorageFs(String root) throws IOException {
		this.root = new File(root);
		if (this.root.exists()) {
			if (!this.root.isDirectory()) {
				throw new IOException();
			}
		} else {
			if (!this.root.mkdirs()) {
				throw new IOException();
			}
		}
	}

	/**
	 * Gets the root folder set with <code>ej.storage.fs.StorageFs.root</code> or {@value #DEFAULT_ROOT_FOLDER} if none.
	 *
	 * @return the root folder.
	 */
	protected static String getRootFolderPropertyOrDefault() {
		return System.getProperty(StorageFs.class.getName() + ROOT_PROPERTY_EXTENSION, DEFAULT_ROOT_FOLDER);
	}

	@Override
	public OutputStream store(String id) throws IOException {
		StorageHelper.checkID(id);
		return new FileOutputStream(getFile(id), false);
	}

	@Override
	public OutputStream modify(String id, int offset) throws IOException {
		StorageHelper.checkID(id);
		StorageHelper.checkOffset(offset);
		return new RandomAccessFileOutputStream(getFile(id), offset);
	}

	/**
	 * Returns the file associated to the given storage id.
	 *
	 * @param id
	 *            the storage identifier.
	 * @return the file associated to the given storage id.
	 */
	protected File getFile(String id) {
		return new File(this.root, id);
	}

	@Override
	public OutputStream append(String id) throws IOException {
		StorageHelper.checkID(id);
		return new FileOutputStream(getFile(id), true);
	}

	@Override
	@Nullable
	public InputStream load(String id) throws IOException {
		StorageHelper.checkID(id);
		File file = getFile(id);
		if (file.exists() && file.isFile()) {
			return new FileInputStream(file);
		} else {
			return null;
		}
	}

	@Override
	public void move(String sourceId, String destinationId) throws IOException {
		StorageHelper.checkID(sourceId);
		StorageHelper.checkID(destinationId);
		File sourceFile = getFile(sourceId);
		if (sourceFile.exists()) {
			File destinationFile = getFile(destinationId);
			if (destinationFile.exists() && !destinationFile.delete()) {
				throw new IOException();
			}
			if (!sourceFile.renameTo(destinationFile)) {
				throw new IOException();
			}
		} else {
			throw new IOException();
		}
	}

	@Override
	public void remove(String id) throws IOException {
		StorageHelper.checkID(id);
		File file = getFile(id);
		if (!file.delete()) {
			throw new IOException();
		}
	}

	@Override
	public long getSize(String id) throws IOException {
		StorageHelper.checkID(id);
		File file = getFile(id);
		if (file.exists() && file.isFile()) {
			return file.length();
		} else {
			throw new IOException();
		}
	}

	@Override
	public boolean exists(String id) throws IOException {
		StorageHelper.checkID(id);
		File file = getFile(id);
		return file.isFile() && file.exists();
	}

	@Override
	public String[] getIds() throws IOException {
		File[] files = this.root.listFiles();
		if (files == null) {
			throw new IOException();
		}
		List<String> idsList = new ArrayList<>(files.length);
		for (File file : files) {
			String name = file.getName();
			if (file.isFile()) {
				idsList.add(name);
			}
		}
		return idsList.toArray(new String[idsList.size()]);
	}
}
