/*
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.image.pathdata;

import java.io.ByteArrayOutputStream;

import ej.microvg.image.ImageGenerator.Format;

/**
 * Encodes a path with two byte arrays: the commands and the commands' attributes. A command is encoded on one byte and
 * the command's attributes are encoded in float.
 */
public class TwoArraysPathData implements PathData {

	private final ByteArrayOutputStream commandsBuffer;
	private final ByteArrayOutputStream coordinatesBuffer;

	/**
	 * Creates an encoder that will encode the path data in two byte arrays.
	 */
	public TwoArraysPathData() {
		this.commandsBuffer = new ByteArrayOutputStream();
		this.coordinatesBuffer = new ByteArrayOutputStream();
	}

	@Override
	public void reset() {
		this.commandsBuffer.reset();
		this.coordinatesBuffer.reset();
	}

	@Override
	public Format getFormat() {
		return Format.VG_FP32;
	}

	@Override
	public void appendCommand(int command) {
		this.commandsBuffer.write((byte) command);
	}

	@Override
	public void appendCoordinate(double coordinate) {
		int i = Float.floatToIntBits((float) coordinate);
		append(this.coordinatesBuffer, i);
	}

	private void append(ByteArrayOutputStream buffer, int i) {
		buffer.write((byte) (i));
		buffer.write((byte) (i >> 8));
		buffer.write((byte) (i >> 16));
		buffer.write((byte) (i >> 24));
	}

	/**
	 * Gets the buffer that contains the encoded command paramerters.
	 *
	 * @return the encoded parameters
	 */
	public byte[] getPathParams() {
		return this.coordinatesBuffer.toByteArray();
	}

	/**
	 * Gets the buffer that contains the encoded commands.
	 *
	 * @return the encoded commands
	 */
	public byte[] getPathCommands() {
		return this.commandsBuffer.toByteArray();
	}
}
