/*
 * Java
 *
 * Copyright 2015 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.observable;

import java.util.ArrayList;
import java.util.List;

/**
 * This class represents an observable object, or "data" in the model-view paradigm. It can be subclassed to represent
 * an object that the application wants to have observed.
 * <p>
 * An observable object can have several observers. An observer may be any object that implements interface
 * {@link Observer}. After an observable instance changes, an application calling the observable's
 * {@link #notifyObservers()} method causes all of its observers to be notified of the change by a call to their
 * {@link Observer#update()} method.
 */
public class Observable {

	private final List<Observer> observers;
	private boolean changed;

	/**
	 * Constructs an observable with no observers.
	 */
	public Observable() {
		this.observers = new ArrayList<>();
	}

	/**
	 * Adds an observer to the set of observers for this object.
	 * <p>
	 * If the observer is already added, nothing changes.
	 * <p>
	 * The order in which notifications will be delivered to multiple observers is not specified.
	 *
	 * @param observer
	 *            an observer to be added.
	 * @throws NullPointerException
	 *             if the parameter is <code>null</code>.
	 */
	public void addObserver(Observer observer) throws NullPointerException {
		if (observer == null) {
			throw new NullPointerException();
		}
		synchronized (this.observers) {
			this.observers.add(observer);
		}
	}

	/**
	 * Deletes an observer from the set of observers of this object.
	 * <p>
	 * If the given observer is <code>null</code> nothing changes.
	 *
	 * @param observer
	 *            the observer to be deleted.
	 */
	public void deleteObserver(Observer observer) {
		synchronized (this.observers) {
			this.observers.remove(observer);
		}
	}

	/**
	 * Marks this observable object as having been changed. The {@link #hasChanged()} method will now return
	 * <code>true</code>.
	 */
	protected void setChanged() {
		this.changed = true;
	}

	/**
	 * Indicates that this object has no longer changed, or that it has already notified all of its observers of its
	 * most recent change, so that the {@link #hasChanged()} method will now return <code>false</code>. This method is
	 * called automatically by the {@link #notifyObservers()} method.
	 */
	protected void clearChanged() {
		this.changed = false;
	}

	/**
	 * Gets whether or not this object has changed.
	 *
	 * @return <code>true</code> if and only if the {@link #setChanged()} method has been called more recently than the
	 *         {@link #clearChanged()} method on this object, <code>false</code> otherwise.
	 */
	public boolean hasChanged() {
		return this.changed;
	}

	/**
	 * Returns the number of observers of this observable object.
	 *
	 * @return the number of observers of this object.
	 */
	public int countObservers() {
		return this.observers.size();
	}

	/**
	 * If this object has changed, as indicated by the {@link #hasChanged()} method, then notify all of its observers
	 * and then call the {@link #clearChanged()} method to indicate that this object has no longer changed.
	 * <p>
	 * Each observer has its update method called.
	 *
	 * @see Observer#update()
	 */
	public void notifyObservers() {
		if (!this.changed) {
			return;
		}
		// create a copy of the observers to avoid calling observers code synchronized with this observable monitor
		Observer[] observers = getObservers();
		clearChanged();
		for (Observer observer : observers) {
			observer.update();
		}
	}

	/**
	 * Gets all observers of this observable object.
	 *
	 * @return an array containing all the observers of this object.
	 */
	public Observer[] getObservers() {
		synchronized (this.observers) {
			return this.observers.toArray(new Observer[this.observers.size()]);
		}
	}

}
