/*
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.android.wearos;

import java.lang.reflect.InvocationTargetException;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.microej.android.MicroEJSupport;
import com.microej.android.application.ApplicationSupport;

import android.view.SurfaceHolder;
import androidx.wear.watchface.ComplicationSlot;
import androidx.wear.watchface.ComplicationSlotsManager;
import androidx.wear.watchface.ListenableWatchFaceService;
import androidx.wear.watchface.TapEvent;
import androidx.wear.watchface.TapType;
import androidx.wear.watchface.WatchFace;
import androidx.wear.watchface.WatchFace.TapListener;
import androidx.wear.watchface.WatchFaceType;
import androidx.wear.watchface.WatchState;
import androidx.wear.watchface.style.CurrentUserStyleRepository;

/**
 * This WatchFaceService implementation can run a MicroEJ Application on Android WearOS using the MicroEJ support
 * engine.
 * <p>
 * The concrete class should implement {@link #getApplicationMainClass()} to provide the main class of the MicroEJ
 * Application. It can also override {@link #runApplication()} to run the Application differently.
 */
public abstract class MicroEJWatchFaceService extends ListenableWatchFaceService {

	private MicroEJCanvasRenderer renderer;

	/**
	 * Returns the main class of the MicroEJ Application.
	 *
	 * @return the main class of the MicroEJ Application.
	 */
	protected abstract String getApplicationMainClass();

	/**
	 * Runs the MicroEJ Application.
	 * <p>
	 * This method invokes the main method of the Application main class (see {@link #getApplicationMainClass()}). This
	 * behavior can be changed by overriding this method.
	 */
	protected void runApplication() {
		String mainClassName = getApplicationMainClass();
		try {
			Class<?> mainClass = Class.forName(mainClassName);
			mainClass.getMethod("main", String[].class).invoke(null, (Object) new String[0]);
		} catch (InvocationTargetException e) {
			e.getCause().printStackTrace();
		} catch (ReflectiveOperationException e) {
			System.err.println("Error invoking main method of main class");
		}
	}

	@Override
	public void onCreate() {
		super.onCreate();

		ApplicationSupport applicationSupport = MicroEJSupport.getApplicationSupport();
		applicationSupport.startup();
		applicationSupport.setContext(this);
	}

	@Override
	protected ListenableFuture<WatchFace> createWatchFaceFuture(SurfaceHolder surfaceHolder, WatchState watchState,
			ComplicationSlotsManager complicationSlotsManager, CurrentUserStyleRepository currentUserStyleRepository) {
		this.renderer = new MicroEJCanvasRenderer(surfaceHolder, currentUserStyleRepository, watchState);

		WatchFace watchface = new WatchFace(WatchFaceType.ANALOG, this.renderer);
		watchface.setTapListener(createTapListener());

		new Thread(new Runnable() {
			@Override
			public void run() {
				runApplication();
			}
		}).start();

		return Futures.immediateFuture(watchface);
	}

	private TapListener createTapListener() {
		return new TapListener() {
			@Override
			public void onTapEvent(int tapType, TapEvent tapEvent, ComplicationSlot complicationSlot) {
				switch (tapType) {
				case TapType.DOWN:
					// touch press but can be canceled
					MicroEJSupport.getApplicationSupport().onPointerPress(MicroEJWatchFaceService.this.renderer,
							tapEvent.getXPos(), tapEvent.getYPos());
					break;
				case TapType.CANCEL:
					// touch canceled (touch event managed by system)
				case TapType.UP:
					// touch released
					MicroEJSupport.getApplicationSupport().onPointerRelease(MicroEJWatchFaceService.this.renderer);
					break;
				default:
					// touch move: not available
					break;
				}
			}
		};
	}
}
