/*
 * Copyright 2022-2023 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.android.wearos;

import java.time.ZonedDateTime;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.microej.android.MicroEJSupport;
import com.microej.android.application.ApplicationDisplay;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Rect;
import android.view.SurfaceHolder;
import androidx.wear.watchface.CanvasType;
import androidx.wear.watchface.ListenableCanvasRenderer2;
import androidx.wear.watchface.Renderer;
import androidx.wear.watchface.WatchState;
import androidx.wear.watchface.style.CurrentUserStyleRepository;
import kotlin.Unit;

/* package */ class MicroEJCanvasRenderer extends ListenableCanvasRenderer2<MicroEJCanvasRenderer.SharedAssets>
implements ApplicationDisplay {

	/* package */ static class SharedAssets implements Renderer.SharedAssets {

		@Override
		public void onDestroy() {
			// do nothing
		}
	}

	private static final long FRAME_PERIOD_MS_DEFAULT = 16L;

	private Bitmap flushBitmap;

	public MicroEJCanvasRenderer(SurfaceHolder surfaceHolder, CurrentUserStyleRepository currentUserStyleRepository,
			WatchState watchState) {
		super(surfaceHolder, currentUserStyleRepository, watchState, CanvasType.HARDWARE, FRAME_PERIOD_MS_DEFAULT);
	}

	@Override
	public ListenableFuture<SharedAssets> createSharedAssetsFuture() {
		return Futures.immediateFuture(new SharedAssets());
	}

	@Override
	public ListenableFuture<Unit> initFuture() {
		MicroEJSupport.getApplicationSupport().setDisplay(this);
		return super.initFuture();
	}

	@Override
	public void onDestroy() {
		MicroEJSupport.getApplicationSupport().unsetDisplay(this);
		super.onDestroy();
	}

	@Override
	public void flush(Bitmap bitmap, Rect bounds) {
		this.flushBitmap = bitmap;
	}

	@Override
	public void render(Canvas canvas, Rect bounds, ZonedDateTime zonedDateTime, SharedAssets sharedAssets) {
		Bitmap flushBitmap = this.flushBitmap;
		if (flushBitmap != null) {
			canvas.drawBitmap(flushBitmap, bounds, bounds, null);
			MicroEJSupport.getApplicationSupport().onDisplayTearing(this);
		}
	}

	@Override
	public void renderHighlightLayer(Canvas canvas, Rect bounds, ZonedDateTime zonedDateTime,
			SharedAssets sharedAssets) {
		// do nothing
	}
}
