/*
 * Java
 *
 * Copyright 2018 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.kf.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;

import ej.kf.Feature;
import ej.kf.Kernel;
import ej.kf.Module;

/**
 * Implementation of a {@link AbstractKFList} optimized for heap size.
 *
 * All internal arrays are resized, and iterators are executed without extra object copy.
 *
 * @see KFList
 */
public class KFList2<T> extends AbstractKFList<T> {

	// Array owned by the Kernel. Sub arrays owned by the context.
	private Object[][] contextsToList;

	public KFList2() {
		contextsToList = new Object[0][];
	}

	@Override
	public synchronized boolean contains(Object element) {
		Module owner = Kernel.getContextOwner();
		Object[][] contextToLists = this.contextsToList;
		int listIndex = getContextListIndex(contextToLists, owner);
		if (listIndex == -1) {
			return false;
		} else {
			Object[] list = contextToLists[listIndex];
			for (Object e : list) {
				if (e.equals(element)) {
					return true;
				}
			}
			return false;
		}
	}

	/**
	 * @throws IllegalArgumentException
	 *             if this method is called by the Kernel and the element is owned by a Feature.
	 */
	@Override
	public synchronized boolean add(T element) {
		Module owner = Kernel.getContextOwner();
		if (owner == Kernel.getInstance() && Kernel.getOwner(element) != owner) {
			throw new IllegalArgumentException(); // TODO P0095F-213
		}
		Object[][] contextToLists = this.contextsToList;
		int listIndex = getContextListIndex(contextToLists, owner);
		if (listIndex == -1) {
			// this is the first time this context adds an element
			Object[] list = new Object[] { element };
			Kernel.enter();
			try {
				int nbContexts = contextToLists.length;
				System.arraycopy(contextToLists, 0, this.contextsToList = new Object[nbContexts + 1][], 0, nbContexts);
				this.contextsToList[nbContexts] = list;
			} finally {
				Kernel.exit();
			}
		} else {
			Object[] list = contextToLists[listIndex];
			int nbElements = list.length;
			System.arraycopy(list, 0, list = new Object[nbElements + 1], 0, nbElements);
			list[nbElements] = element;
			Kernel.enter();
			try {
				contextToLists[listIndex] = list;
			} finally {
				Kernel.exit();
			}
		}
		return true;
	}

	@Override
	public synchronized boolean remove(Object element) {
		Module owner = Kernel.getContextOwner();
		if (owner == Kernel.getInstance() && Kernel.getOwner(element) != owner) {
			throw new IllegalArgumentException(); // TODO P0095F-213
		}
		Object[][] contextToLists = this.contextsToList;
		int listIndex = getContextListIndex(contextToLists, owner);
		if (listIndex == -1) {
			return false;
		}

		Object[] list = contextToLists[listIndex];
		int elementIndex = -1;
		int nbElements = list.length;
		for (int i = 0; i < nbElements; ++i) {
			if (list[i].equals(element)) {
				elementIndex = i;
				break;
			}
		}

		if (elementIndex == -1) {
			return false;
		}

		if (nbElements == 1) {
			// list will be empty - resize contextToList
			Kernel.enter();
			try {
				this.contextsToList = removeContextListIndex(contextToLists, listIndex);
			} finally {
				Kernel.exit();
			}
		} else {
			Object[] newList = new Object[nbElements - 1];
			copyExceptIndex(list, elementIndex, newList);
			Kernel.enter();
			try {
				this.contextsToList[listIndex] = newList;
			} finally {
				Kernel.exit();
			}
		}
		return true;
	}

	@Override
	public Iterator<T> iterator() {
		Module owner = Kernel.getContextOwner();
		Object[][] contextToLists = this.contextsToList;
		if (owner == Kernel.getInstance()) {
			return new ContextListIterator(contextToLists);
		} else {
			int listIndex = getContextListIndex(contextToLists, owner);
			if (listIndex == -1) {
				return new ArrayList<T>(0).iterator();
			}
			return new ListIterator(contextToLists[listIndex]);
		}
	}

	@Override
	public synchronized boolean addAll(Collection<? extends T> c) {
		boolean result = false;
		for (T e : c) {
			result |= add(e);
		}
		return result;
	}

	@Override
	public synchronized void clear() {
		Module owner = Kernel.getContextOwner();
		if (owner == Kernel.getInstance()) {
			Kernel.enter();
			try {
				this.contextsToList = new Object[0][];
			} finally {
				Kernel.exit();
			}
		} else {
			Object[][] contextToLists = this.contextsToList;
			int listIndex = getContextListIndex(contextToLists, owner);
			if (listIndex == -1) {
				return;
			}
			Kernel.enter();
			try {
				this.contextsToList = removeContextListIndex(contextToLists, listIndex);
			} finally {
				Kernel.exit();
			}
		}
	}

	@Override
	public synchronized boolean isEmpty() {
		Module owner = Kernel.getContextOwner();
		Object[][] contextToLists = this.contextsToList;
		if (owner == Kernel.getInstance()) {
			return contextToLists.length == 0;
		} else {
			int listIndex = getContextListIndex(contextToLists, owner);
			if (listIndex == -1) {
				return true;
			}

			Object[] list = contextToLists[listIndex];
			return list.length == 0;
		}
	}

	@Override
	public synchronized int size() {
		Module owner = Kernel.getContextOwner();
		if (owner == Kernel.getInstance()) {
			int size = 0;
			for (Object[] list : contextsToList) {
				size += list.length;
			}
			return size;
		} else {
			Object[][] contextToLists = this.contextsToList;
			int listIndex = getContextListIndex(contextToLists, owner);
			if (listIndex == -1) {
				return 0;
			} else {
				Object[] list = contextToLists[listIndex];
				return list.length;
			}
		}
	}

	@Override
	protected void toArray(ArrayList<T> allList) {
		Object[][] contextToLists = this.contextsToList;
		if (Kernel.isInKernelMode()) {
			for (Object[] list : contextToLists) {
				addAll(allList, list);
			}
		} else {
			Module owner = Kernel.getContextOwner();
			int listIndex = getContextListIndex(contextToLists, owner);
			if (listIndex != -1) {
				Object[] list = contextToLists[listIndex];
				addAll(allList, list);
			}
		}
	}

	@Override
	protected void removeAllElementsOwnedBy(Feature feature) {
		Object[][] contextToLists = this.contextsToList;
		int listIndex = getContextListIndex(contextToLists, feature);
		if (listIndex != -1) {
			this.contextsToList = removeContextListIndex(contextToLists, listIndex);
		}
	}

	/**
	 * Get the element index in the given list.
	 *
	 * @return the index in the list, -1 if not found.
	 */
	private static int getContextListIndex(Object[][] list, Module context) {
		Kernel.enter(); // loop on an heterogeneous object list
		int length = list.length;
		for (int i = 0; i < length; ++i) {
			if (Kernel.getOwner(list[i]) == context) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Create a new array and copy the content except the element at the given index
	 */
	private Object[][] removeContextListIndex(Object[][] contextToLists, int listIndex) {
		Object[][] newContextsToList = new Object[contextToLists.length - 1][];
		copyExceptIndex(contextToLists, listIndex, newContextsToList);
		return newContextsToList;
	}

	/**
	 * Copy the content of the first array to the second array, except the element at the given index
	 */
	private static <T> void copyExceptIndex(T[] from, int index, T[] to) {
		if (index > 0) {
			System.arraycopy(from, 0, to, 0, index);
		}
		int length = from.length;
		if (index < length - 1) {
			System.arraycopy(from, index + 1, to, index, length - index - 1);
		}
	}

	/**
	 * Add all elements to the given {@link ArrayList}, starting at the given index.
	 */
	@SuppressWarnings("unchecked")
	private static <T> void addAll(ArrayList<T> allList, Object[] list) {
		for (int i = 0; i < list.length; ++i) {
			allList.add((T) list[i]);
		}
	}

	class ContextListIterator implements Iterator<T> {

		private final Object[][] contextToLists;
		private Iterator<T> currentIterator; // null when not computed or when end of the composite iterator is reached
		private int nextIteratorIndex; // 0 based. Position of the next iterator.

		public ContextListIterator(Object[][] contextToLists) {
			this.contextToLists = contextToLists;
			this.nextIteratorIndex = 0;
		}

		@Override
		public boolean hasNext() {
			// Implementation Note: when this method returns true, currentIterator is not null for sure
			int nbIterators = contextToLists.length;
			while (nextIteratorIndex < nbIterators) {
				if (currentIterator == null) {
					currentIterator = new ListIterator(contextToLists[nextIteratorIndex]);
				}
				if (currentIterator.hasNext()) {
					return true;
				}
				currentIterator = null;
				++nextIteratorIndex;
			}
			return false;
		}

		@Override
		public T next() {
			if (hasNext()) {
				return currentIterator.next();
			}
			throw new NoSuchElementException();
		}

		@Override
		public void remove() {
			if (currentIterator == null) {
				throw new IllegalStateException();
			}
			currentIterator.remove();
		}
	}

	class ListIterator implements Iterator<T> {

		private final Object[] list;
		private int listPtr; // index of the last returned element.

		public ListIterator(Object[] list) {
			this.list = list;
			this.listPtr = -1;
		}

		@Override
		public boolean hasNext() {
			return listPtr < list.length - 1;
		}

		@SuppressWarnings("unchecked")
		@Override
		public T next() {
			return (T) list[++listPtr];
		}

		@Override
		public void remove() {
			throw new UnsupportedOperationException();
		}
	}
}
