/*
 * Java
 *
 * Copyright 2010-2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.event.generator;

/**
 * Keypad is a Keyboard that defines an event generator for 12-key keypads. It follows the ETSI ES 202 130 mapping,
 * which takes into account ETSI, ITU-T, CEN and ISO/IEC specifications and recommendations. Also see ISO/IEC 10646.<br>
 * <p>
 * The key mapping is defined in Table 33 and in Table 63 of ETSI ES 202 130 (v1.1.1). In addition the next three keys
 * have extended mapping defined as:<br>
 * key10: <code>'*'</code> : this key is only used to switch from one mode to another<br>
 * key11: <code>' '</code>, <code>'+'</code>, <code>'0'</code> in order<br>
 * key12: <code>'\n'</code>, <code>'#'</code> in order<br>
 * <p>
 * Keypad sends low-level Keyboard events with basic code of the key (from '0' to '9', '#' or '*') and high level
 * {@link #TEXT_INPUT} events with next key code mapping until key is validated (Key codes are scrolled in order,
 * circularly). A key is validated when no new key has been pressed before validation delay or if an other physical key
 * of the keypad is pressed. The delay starts when the key is pressed, so a key may be validated even if it is not yet
 * released. When a key is validated Keypad sends {@link #KEY_VALIDATED} event.
 *
 * The delay for key validation can be modified at any time using {@link #setDelay(int)} Keypad uses 4 different modes
 * to filter the letters that are scrolled.
 * <ul>
 * <li>{@link #NUM}: only digits are selected</li>
 * <li>{@link #ALPHA}: digits and letters are selected</li>
 * <li>{@link #CAP}: only capital letters and digits are selected</li>
 * <li>{@link #CAP1}: same as {@link #CAP}, but must switch to {@link #ALPHA} mode after the first character is
 * validated</li>
 * </ul>
 * For example, assuming that low-level events are enabled (see {@link Keyboard}) pressing the '2' key twice rapidly and
 * then waiting a little amount of time after activation delay will generate:
 * <ul>
 * <li>{@link #KEY_DOWN} '2'</li>
 * <li>{@link #TEXT_INPUT} 'a'</li>
 * <li>{@link #KEY_UP} '2'</li>
 * <li>{@link #KEY_DOWN} '2'</li>
 * <li>{@link #TEXT_INPUT} 'b'</li>
 * <li>{@link #KEY_UP} '2'</li>
 * <li>{@link #KEY_VALIDATED} (after a delay, see {@link #setDelay(int)})</li>
 * </ul>
 *
 * @see Keyboard
 */
public abstract class Keypad extends Keyboard {

	/**
	 * The KEY_VALIDATED event action.<br>
	 * This event action is sent when last key is validated, meaning that no new key has been pressed during delay. <br>
	 *
	 * @see Keyboard#getAction(int)
	 */
	public static final int KEY_VALIDATED = 8;

	/**
	 * The ALPHA mode.<br>
	 * In ALPHA mode the keypad may return several letters and digits according to the number of consequent key press.
	 * <br>
	 *
	 * @see #setMode(int)
	 */
	public static final int ALPHA = 0;

	/**
	 * The NUM mode.<br>
	 * In NUM mode the keypad may return several digits according to the number of consequent key press. <br>
	 *
	 * @see #setMode(int)
	 */
	public static final int NUM = 1;

	/**
	 * The CAP mode.<br>
	 * In CAP mode only capital letters and digits are returned <br>
	 *
	 * @see #setMode(int)
	 */
	public static final int CAP = 2;

	/**
	 * The CAP1 mode.<br>
	 * In CAP1 mode is the same has CAP mode, but automatically switch to ALPHA mode after the {@link #KEY_UP} event
	 * <br>
	 *
	 * @see #CAP
	 * @see #setMode(int)
	 */
	public static final int CAP1 = 3;

	/**
	 * Keypads hold a buffer (potentially of size one) that stores the keys before they are used by the application (key
	 * associated to {@link #KEY_UP}, {@link #KEY_DOWN} and {@link #TEXT_INPUT} event).
	 *
	 * @param size
	 *            of the buffer.
	 * @see Keyboard#dropOnFull()
	 */
	public Keypad(int size) {
		super(size);
	}

	/**
	 * @return Event.KEYPAD
	 */

	@Override
	public int getEventType() {
		throw new RuntimeException();
	}

	/**
	 * Sets the delay The delay is the value between two press that allows to select a letter out of several. The
	 * default value is 750ms.
	 *
	 * @param delay
	 *            the delay to set
	 * @throws IllegalArgumentException
	 *             if delay is negative or zero
	 */
	public void setDelay(int delay) {
		throw new RuntimeException();
	}

	/**
	 * Gets the delay. The delay is the value between two press that allows to select a letter out of several. The
	 * default value is 750ms.
	 *
	 * @return the delay
	 */
	public int getDelay() {
		throw new RuntimeException();
	}

	/**
	 * Sets the mode.
	 *
	 * @param mode
	 *            one of {@link #ALPHA}, {@link #NUM}, {@link #CAP}, {@link #CAP1}
	 * @throws IllegalArgumentException
	 *             if mode is not one of {@link #ALPHA}, {@link #NUM}, {@link #CAP}, {@link #CAP1}
	 */
	public void setMode(int mode) {
		throw new RuntimeException();
	}

	/**
	 * Gets the mode.
	 *
	 * @return one of {@link #ALPHA}, {@link #NUM}, {@link #CAP}, {@link #CAP1}
	 */
	public int getMode() {
		throw new RuntimeException();
	}

	/**
	 * Sets the language. Select the nearest mapping available on the platform.<br>
	 * The language must be a valid ISO language code. These codes are the lower-case, two-letter codes as defined by
	 * ISO-639.
	 *
	 * @param language
	 *            the ISO language code
	 */
	public abstract void setLanguage(String language);

	/**
	 * Gets the currently selected language.
	 *
	 * @return the ISO language code
	 * @see #setLanguage(String)
	 */
	public abstract String getLanguage();

	/**
	 * Gets the supported languages for this keypad.
	 *
	 * @return an array of supported ISO language codes
	 */
	public abstract String[] getSupportedLanguages();

	/**
	 * Gets the complete array of chars associated with the specified key. The <code>key</code> must be one of
	 * <code>'0'</code>, <code>'1'</code>, <code>'2'</code>, <code>'3'</code>, <code>'4'</code>, <code>'5'</code>,
	 * <code>'6'</code>, <code>'7'</code>, <code>'8'</code>, <code>'9'</code>, <code>'*'</code>, <code>'0'</code>,
	 * <code>'#'</code>.
	 *
	 * @param key
	 *            the key
	 * @return the array of char associated with the <code>key</code>
	 */
	public abstract char[] getAssignment(char key);

}
