/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.dimension;

import ej.annotation.Nullable;
import ej.mwt.Widget;
import ej.mwt.util.Rectangle;
import ej.mwt.util.Size;

/**
 * Represents a dimension constraint for a widget.
 */
public interface Dimension {

	/**
	 * Gets the available size for a widget.
	 * <p>
	 * The given available size is the available width and height minus the outlines.
	 * <p>
	 * The given size is updated by this method to set the available size for this dimension.
	 *
	 * @param widget
	 *            the widget.
	 * @param availableWidth
	 *            the available width.
	 * @param availableHeight
	 *            the available height.
	 * @param availableSize
	 *            the available size.
	 */
	void getAvailableSize(Widget widget, int availableWidth, int availableHeight, Size availableSize);

	/**
	 * Computes the optimal size of a widget.
	 * <p>
	 * The given optimal size is the optimal size of the widget considering the available size.
	 * <p>
	 * The given size is updated by this method to set the optimal size for this dimension.
	 *
	 * @param widget
	 *            the widget.
	 * @param availableWidth
	 *            the available width.
	 * @param availableHeight
	 *            the available height.
	 * @param optimalSize
	 *            the optimal size.
	 */
	void computeOptimalSize(Widget widget, int availableWidth, int availableHeight, Size optimalSize);

	/**
	 * Lays out a widget.
	 * <p>
	 * The style of the widget must be set before (not <code>null</code>).
	 * <p>
	 * This dimension is applied to compute the widget bounds and align the widget in the given bounds. The given
	 * rectangle is updated with the updated bounds.
	 *
	 * @param widget
	 *            the widget.
	 * @param bounds
	 *            the bounds of the widget.
	 */
	void layOut(Widget widget, Rectangle bounds);

	@Override
	boolean equals(@Nullable Object obj);

	@Override
	int hashCode();
}
