/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.observable;

import ej.annotation.Nullable;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.observable.Observable;
import ej.observable.Observer;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;

/**
 * Extension of {@link Observable} which can handle observers from any {@link Module}.
 */
public class KernelObservable extends Observable {

	private @Nullable FeatureStateListener featureStateListener;

	/**
	 * Creates a kernel observable.
	 * <p>
	 * This registers a feature state listener in the kernel. The {@link #dispose()} method has to be called once this
	 * object is not used anymore in order to unregister the feature state listener.
	 */
	public KernelObservable() {
		if (Kernel.getContextOwner() != Kernel.getInstance()) {
			throw new IllegalAccessError();
		}

		this.featureStateListener = new FeatureStateListener() {
			@Override
			public void stateChanged(@Nullable Feature feature, @Nullable State previousState) {
				if (feature != null && feature.getState() == State.STOPPED) {
					deleteObservers(feature);
				}
			}
		};
		Kernel.addFeatureStateListener(this.featureStateListener);
	}

	/**
	 * Disposes this kernel observable. Once it is disposed, observers can not be registered anymore.
	 */
	public void dispose() {
		FeatureStateListener featureStateListener = this.featureStateListener;
		if (featureStateListener != null) {
			deleteObservers();
			Kernel.removeFeatureStateListener(featureStateListener);
			this.featureStateListener = null;
		}
	}

	/**
	 * {@inheritDoc}
	 *
	 * @throws IllegalStateException
	 *             if this kernel observable has been disposed.
	 */
	@Override
	public void addObserver(Observer observer) {
		if (this.featureStateListener == null) {
			throw new IllegalStateException();
		}
		Kernel.enter();
		super.addObserver(observer);
	}

	/**
	 * Deletes every observer which belongs to the given module from the set of observers of this kernel observable.
	 *
	 * @param module
	 *            the module which may own observers.
	 */
	private void deleteObservers(Module module) {
		Observer[] observers = getObservers();
		for (Observer observer : observers) {
			assert (observer != null);
			if (Kernel.getOwner(observer) == module) {
				deleteObserver(observer);
			}
		}
	}
}
