/*
 * Java
 *
 * Copyright 2016-2024 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 */
package org.junit;

/**
 * Thrown when an {@link org.junit.Assert#assertEquals(Object, Object)
 * assertEquals(String, String)} fails. Create and throw a
 * <code>ComparisonFailure</code> manually if you want to show users the
 * difference between two complex strings.
 * <p>
 * Inspired by a patch from Alex Chaffee (alex@purpletech.com)
 *
 * @since 4.0
 */
public class ComparisonFailure extends AssertionError {
	/**
	 * The maximum length for expected and actual strings. If it is exceeded, the
	 * strings should be shortened.
	 *
	 * @see ComparisonCompactor
	 */
	private static final int MAX_CONTEXT_LENGTH = 20;
	private static final long serialVersionUID = 1L;

	/*
	 * We have to use the f prefix until the next major release to ensure
	 * serialization compatibility. See
	 * https://github.com/junit-team/junit/issues/976
	 */
	private final String fExpected;
	private final String fActual;

	/**
	 * Constructs a comparison failure.
	 *
	 * @param message
	 *            the identifying message or null
	 * @param expected
	 *            the expected string value
	 * @param actual
	 *            the actual string value
	 */
	public ComparisonFailure(String message, String expected, String actual) {
		super(message);
		this.fExpected = expected;
		this.fActual = actual;
	}

	/**
	 * Returns "..." in place of common prefix and "..." in place of common suffix
	 * between expected and actual.
	 *
	 * @see Throwable#getMessage()
	 */
	@Override
	public String getMessage() {
		return new ComparisonCompactor(MAX_CONTEXT_LENGTH, this.fExpected, this.fActual).compact(super.getMessage());
	}

	/**
	 * Returns the actual string value
	 *
	 * @return the actual string value
	 */
	public String getActual() {
		return this.fActual;
	}

	/**
	 * Returns the expected string value
	 *
	 * @return the expected string value
	 */
	public String getExpected() {
		return this.fExpected;
	}

	private static class ComparisonCompactor {
		private static final String ELLIPSIS = "...";
		private static final String DIFF_END = "]";
		private static final String DIFF_START = "[";

		/**
		 * The maximum length for <code>expected</code> and <code>actual</code> strings
		 * to show. When <code>contextLength</code> is exceeded, the Strings are
		 * shortened.
		 */
		private final int contextLength;
		private final String expected;
		private final String actual;

		/**
		 * @param contextLength
		 *            the maximum length of context surrounding the difference between
		 *            the compared strings. When context length is exceeded, the
		 *            prefixes and suffixes are compacted.
		 * @param expected
		 *            the expected string value
		 * @param actual
		 *            the actual string value
		 */
		public ComparisonCompactor(int contextLength, String expected, String actual) {
			this.contextLength = contextLength;
			this.expected = expected;
			this.actual = actual;
		}

		public String compact(String message) {
			if (this.expected == null || this.actual == null || this.expected.equals(this.actual)) {
				return Assert.format(message, this.expected, this.actual);
			} else {
				DiffExtractor extractor = new DiffExtractor();
				String compactedPrefix = extractor.compactPrefix();
				String compactedSuffix = extractor.compactSuffix();
				return Assert.format(message, compactedPrefix + extractor.expectedDiff() + compactedSuffix,
						compactedPrefix + extractor.actualDiff() + compactedSuffix);
			}
		}

		private String sharedPrefix() {
			int end = Math.min(this.expected.length(), this.actual.length());
			for (int i = 0; i < end; i++) {
				if (this.expected.charAt(i) != this.actual.charAt(i)) {
					return this.expected.substring(0, i);
				}
			}
			return this.expected.substring(0, end);
		}

		private String sharedSuffix(String prefix) {
			int suffixLength = 0;
			int maxSuffixLength = Math.min(this.expected.length() - prefix.length(),
					this.actual.length() - prefix.length()) - 1;
			for (; suffixLength <= maxSuffixLength; suffixLength++) {
				if (this.expected.charAt(this.expected.length() - 1 - suffixLength) != this.actual
						.charAt(this.actual.length() - 1 - suffixLength)) {
					break;
				}
			}
			return this.expected.substring(this.expected.length() - suffixLength);
		}

		private class DiffExtractor {
			private final String sharedPrefix;
			private final String sharedSuffix;

			/**
			 * Can not be instantiated outside
			 * {@link org.junit.ComparisonFailure.ComparisonCompactor}.
			 */
			private DiffExtractor() {
				this.sharedPrefix = sharedPrefix();
				this.sharedSuffix = sharedSuffix(this.sharedPrefix);
			}

			public String expectedDiff() {
				return extractDiff(ComparisonCompactor.this.expected);
			}

			public String actualDiff() {
				return extractDiff(ComparisonCompactor.this.actual);
			}

			public String compactPrefix() {
				if (this.sharedPrefix.length() <= ComparisonCompactor.this.contextLength) {
					return this.sharedPrefix;
				}
				return ELLIPSIS + this.sharedPrefix
						.substring(this.sharedPrefix.length() - ComparisonCompactor.this.contextLength);
			}

			public String compactSuffix() {
				if (this.sharedSuffix.length() <= ComparisonCompactor.this.contextLength) {
					return this.sharedSuffix;
				}
				return this.sharedSuffix.substring(0, ComparisonCompactor.this.contextLength) + ELLIPSIS;
			}

			private String extractDiff(String source) {
				return DIFF_START
						+ source.substring(this.sharedPrefix.length(), source.length() - this.sharedSuffix.length())
						+ DIFF_END;
			}
		}
	}
}
