/*
 * Protocol Buffers - Google's data interchange format
 * Copyright 2008 Google Inc.  All rights reserved.
 * https: *developers.google.com/protocol-buffers/
 * Copyright 2020 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.google.protobuf;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * A table of known extensions, searchable by name or field number. When parsing a protocol message that might have
 * extensions, you must provide an {@code ExtensionRegistryLite} in which you have registered any extensions that you
 * want to be able to parse. Otherwise, those extensions will just be treated like unknown fields.
 *
 * <p>
 * For example, if you had the {@code .proto} file:
 *
 * <pre>
 * option java_class = "MyProto";
 *
 * message Foo {
 *   extensions 1000 to max;
 * }
 *
 * extend Foo {
 *   optional int32 bar;
 * }
 * </pre>
 *
 * Then you might write code like:
 *
 * <pre>
 * ExtensionRegistryLite registry = ExtensionRegistryLite.newInstance();
 * registry.add(MyProto.bar);
 * MyProto.Foo message = MyProto.Foo.parseFrom(input, registry);
 * </pre>
 *
 * <p>
 * Background:
 *
 * <p>
 * You might wonder why this is necessary. Two alternatives might come to mind. First, you might imagine a system where
 * generated extensions are automatically registered when their containing classes are loaded. This is a popular
 * technique, but is bad design; among other things, it creates a situation where behavior can change depending on what
 * classes happen to be loaded. It also introduces a security vulnerability, because an unprivileged class could cause
 * its code to be called unexpectedly from a privileged class by registering itself as an extension of the right type.
 *
 * <p>
 * Another option you might consider is lazy parsing: do not parse an extension until it is first requested, at which
 * point the caller must provide a type to use. This introduces a different set of problems. First, it would require a
 * mutex lock any time an extension was accessed, which would be slow. Second, corrupt data would not be detected until
 * first access, at which point it would be much harder to deal with it. Third, it could violate the expectation that
 * message objects are immutable, since the type provided could be any arbitrary message class. An unprivileged user
 * could take advantage of this to inject a mutable object into a message belonging to privileged code and create
 * mischief.
 *
 * @author kenton@google.com Kenton Varda
 */
public class ExtensionRegistryLite {

	// Set true to enable lazy parsing feature for MessageSet.
	//
	// TODO(xiangl): Now we use a global flag to control whether enable lazy
	// parsing feature for MessageSet, which may be too crude for some
	// applications. Need to support this feature on smaller granularity.
	private static volatile boolean eagerlyParseMessageSets = false;

	// Visible for testing.
	static final String EXTENSION_CLASS_NAME = "com.google.protobuf.Extension";

	/* @Nullable */
	static Class<?> resolveExtensionClass() {
		try {
			return Class.forName(EXTENSION_CLASS_NAME);
		} catch (ClassNotFoundException e) {
			return null;
		}
	}

	/* @Nullable */
	private static final Class<?> extensionClass = resolveExtensionClass();

	public static boolean isEagerlyParseMessageSets() {
		return eagerlyParseMessageSets;
	}

	public static void setEagerlyParseMessageSets(boolean isEagerlyParse) {
		eagerlyParseMessageSets = isEagerlyParse;
	}

	/**
	 * Construct a new, empty instance.
	 *
	 * <p>
	 * This may be an {@code ExtensionRegistry} if the full (non-Lite) proto libraries are available.
	 */
	public static ExtensionRegistryLite newInstance() {
		return new ExtensionRegistryLite();
	}

	/**
	 * Get the unmodifiable singleton empty instance of either ExtensionRegistryLite or {@code ExtensionRegistry} (if
	 * the full (non-Lite) proto libraries are available).
	 */
	public static ExtensionRegistryLite getEmptyRegistry() {
		return EMPTY_REGISTRY_LITE;
	}

	/** Returns an unmodifiable view of the registry. */
	public ExtensionRegistryLite getUnmodifiable() {
		return new ExtensionRegistryLite(this);
	}

	/**
	 * Find an extension by containing type and field number.
	 *
	 * @return Information about the extension if found, or {@code null} otherwise.
	 */
	@SuppressWarnings("unchecked")
	public <ContainingType extends MessageLite> GeneratedMessageLite.GeneratedExtension<ContainingType, ?> findLiteExtensionByNumber(
			final ContainingType containingTypeDefaultInstance, final int fieldNumber) {
		return (GeneratedMessageLite.GeneratedExtension<ContainingType, ?>) this.extensionsByNumber
				.get(new ObjectIntPair(containingTypeDefaultInstance, fieldNumber));
	}

	/** Add an extension from a lite generated file to the registry. */
	public final void add(final GeneratedMessageLite.GeneratedExtension<?, ?> extension) {
		this.extensionsByNumber
				.put(new ObjectIntPair(extension.getContainingTypeDefaultInstance(), extension.getNumber()), extension);
	}

	// =================================================================
	// Private stuff.

	// Constructors are package-private so that ExtensionRegistry can subclass
	// this.

	ExtensionRegistryLite() {
		this.extensionsByNumber = new HashMap<ObjectIntPair, GeneratedMessageLite.GeneratedExtension<?, ?>>();
	}

	static final ExtensionRegistryLite EMPTY_REGISTRY_LITE = new ExtensionRegistryLite(true);

	ExtensionRegistryLite(ExtensionRegistryLite other) {
		if (other == EMPTY_REGISTRY_LITE) {
			this.extensionsByNumber = Collections.emptyMap();
		} else {
			this.extensionsByNumber = Collections.unmodifiableMap(other.extensionsByNumber);
		}
	}

	private final Map<ObjectIntPair, GeneratedMessageLite.GeneratedExtension<?, ?>> extensionsByNumber;

	ExtensionRegistryLite(boolean empty) {
		this.extensionsByNumber = Collections.emptyMap();
	}

	/** A (Object, int) pair, used as a map key. */
	private static final class ObjectIntPair {
		private final Object object;
		private final int number;

		ObjectIntPair(final Object object, final int number) {
			this.object = object;
			this.number = number;
		}

		@Override
		public int hashCode() {
			return System.identityHashCode(this.object) * ((1 << 16) - 1) + this.number;
		}

		@Override
		public boolean equals(final Object obj) {
			if (!(obj instanceof ObjectIntPair)) {
				return false;
			}
			final ObjectIntPair other = (ObjectIntPair) obj;
			return this.object == other.object && this.number == other.number;
		}
	}
}
