/*
 * Java
 *
 * Copyright 2019-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.library.iot.rcommand.bluetooth;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeoutException;

import ej.annotation.Nullable;
import ej.library.iot.rcommand.bluetooth.commands.AddServiceCommand;
import ej.library.iot.rcommand.bluetooth.commands.ConnectCommand;
import ej.library.iot.rcommand.bluetooth.commands.DisableCommand;
import ej.library.iot.rcommand.bluetooth.commands.DisconnectCommand;
import ej.library.iot.rcommand.bluetooth.commands.DiscoverServicesCommand;
import ej.library.iot.rcommand.bluetooth.commands.EnableCommand;
import ej.library.iot.rcommand.bluetooth.commands.PairCommand;
import ej.library.iot.rcommand.bluetooth.commands.PairReplyCommand;
import ej.library.iot.rcommand.bluetooth.commands.PasskeyReplyCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendNotificationCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendReadRequestCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendReadResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendWriteRequestCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendWriteResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.StartAdvertisingCommand;
import ej.library.iot.rcommand.bluetooth.commands.StartScanningCommand;
import ej.library.iot.rcommand.bluetooth.commands.StopAdvertisingCommand;
import ej.library.iot.rcommand.bluetooth.commands.StopScanningCommand;
import ej.library.iot.rcommand.bluetooth.notifications.EventNotificationListener;
import ej.rcommand.synchronous.Command;
import ej.rcommand.synchronous.RemoteCommandClient;

public class BluetoothHost {

	private static final int ENABLE_COMMAND_TIMEOUT = 5_000;
	private static final int DEFAULT_COMMAND_TIMEOUT = 1_000;
	private static final int DEFAULT_EVENT_QUEUE_SIZE = 10;

	private static final BluetoothHost INSTANCE = new BluetoothHost();

	private final List<byte[]> eventQueue;

	private @Nullable RemoteCommandClient rcommandClient;
	private long commandTimeout;
	private int eventQueueSize;
	private boolean enabled;

	private BluetoothHost() {
		this.eventQueue = new LinkedList<>();
		this.rcommandClient = null;
		this.commandTimeout = DEFAULT_COMMAND_TIMEOUT;
		this.eventQueueSize = DEFAULT_EVENT_QUEUE_SIZE;
		this.enabled = false;
	}

	public static BluetoothHost getInstance() {
		return INSTANCE;
	}

	public void setup(RemoteCommandClient rcommandClient, long commandTimeout, int eventQueueSize) {
		rcommandClient.registerNotificationListener(new EventNotificationListener());

		this.rcommandClient = rcommandClient;
		this.commandTimeout = commandTimeout;
		this.eventQueueSize = eventQueueSize;
	}

	public void pushEvent(byte[] event) {
		if (event.length > 0) {
			synchronized (this.eventQueue) {
				while (this.eventQueue.size() >= this.eventQueueSize) {
					System.out.println("Warning: Bluetooth event discarded due to queue being full");
					this.eventQueue.remove(0);
				}
				this.eventQueue.add(event);
				this.eventQueue.notifyAll();
			}
		}
	}

	public int waitEvent(byte[] buffer, int bufferLength) {
		synchronized (this.eventQueue) {
			while (this.enabled) {
				if (!this.eventQueue.isEmpty()) {
					byte[] event = this.eventQueue.remove(0);
					if (event.length > bufferLength) {
						System.out.println("Warning: Bluetooth event discarded due to buffer being too small");
					} else {
						System.arraycopy(event, 0, buffer, 0, event.length);
						return event.length;
					}
				} else {
					try {
						this.eventQueue.wait();
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}
			}
		}

		return 0;
	}

	public boolean enable() {
		this.enabled = sendBooleanCommand(new EnableCommand(), ENABLE_COMMAND_TIMEOUT);
		return this.enabled;
	}

	public void disable() {
		sendCommand(new DisableCommand(), this.commandTimeout);

		this.enabled = false;

		synchronized (this.eventQueue) {
			this.eventQueue.clear();
			this.eventQueue.notifyAll();
		}
	}

	public boolean startScanning(byte filterAction, byte filterType, byte[] filterData, int filterDataSize) {
		return sendBooleanCommand(new StartScanningCommand(filterAction, filterType, filterData, filterDataSize));
	}

	public boolean stopScanning() {
		return sendBooleanCommand(new StopScanningCommand());
	}

	public boolean startAdvertising(byte[] advertisementData, int advertisementDataSize) {
		return sendBooleanCommand(new StartAdvertisingCommand(advertisementData, advertisementDataSize));
	}

	public boolean stopAdvertising() {
		return sendBooleanCommand(new StopAdvertisingCommand());
	}

	public boolean connect(byte[] deviceAddress) {
		return sendBooleanCommand(new ConnectCommand(deviceAddress));
	}

	public boolean disconnect(short connHandle) {
		return sendBooleanCommand(new DisconnectCommand(connHandle));
	}

	public boolean pair(short connHandle) {
		return sendBooleanCommand(new PairCommand(connHandle));
	}

	public boolean pairReply(short connHandle, boolean accept) {
		return sendBooleanCommand(new PairReplyCommand(connHandle, accept));
	}

	public boolean passkeyReply(short connHandle, boolean accept, int passkey) {
		return sendBooleanCommand(new PasskeyReplyCommand(connHandle, accept, passkey));
	}

	public boolean discoverServices(short connHandle, @Nullable byte[] uuid) {
		return sendBooleanCommand(new DiscoverServicesCommand(connHandle, uuid));
	}

	public boolean addService(byte[] service, short[] handles) {
		int numAttr = service[16] + service[17];
		int serviceSize = 18 + numAttr * 20;
		return sendBooleanCommand(new AddServiceCommand(service, serviceSize, handles));
	}

	public boolean sendReadRequest(short connHandle, short attributeHandle) {
		return sendBooleanCommand(new SendReadRequestCommand(connHandle, attributeHandle));
	}

	public boolean sendWriteRequest(short connHandle, short attributeHandle, byte[] value, int valueSize,
			boolean noResponse) {
		return sendBooleanCommand(
				new SendWriteRequestCommand(connHandle, attributeHandle, value, valueSize, noResponse));
	}

	public boolean sendReadResponse(short connHandle, short attributeHandle, byte status, byte[] value, int valueSize) {
		return sendBooleanCommand(new SendReadResponseCommand(connHandle, attributeHandle, status, value, valueSize));
	}

	public boolean sendWriteResponse(short connHandle, short attributeHandle, byte status) {
		return sendBooleanCommand(new SendWriteResponseCommand(connHandle, attributeHandle, status));
	}

	public boolean sendNotification(short connHandle, short attributeHandle, byte[] value, int valueSize,
			boolean confirm) {
		return sendBooleanCommand(new SendNotificationCommand(connHandle, attributeHandle, value, valueSize, confirm));
	}

	private @Nullable <T> T sendCommand(Command<T> command, long timeout) {
		RemoteCommandClient rcommandClient = this.rcommandClient;
		if (rcommandClient == null) {
			return null;
		}

		try {
			return rcommandClient.execute(command, timeout);
		} catch (IOException | InterruptedException | TimeoutException e) {
			System.out.println("Error: error executing command " + command.getName() + ": " + e.toString());
			return null;
		}
	}

	private boolean sendBooleanCommand(Command<Boolean> command, long timeout) {
		Boolean result = sendCommand(command, timeout);
		return (result != null && result.booleanValue());
	}

	private boolean sendBooleanCommand(Command<Boolean> command) {
		return sendBooleanCommand(command, this.commandTimeout);
	}
}
