/*
 * Java
 *
 * Copyright 2021-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsException;

/**
 * Test the loading of fonts.
 */
public class TestFontLoading {

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Test that loading an ttf that does not exist in the classpath throws the correct exception.
	 */
	@Test(expected = VectorGraphicsException.class)
	public void testNonExistingFileException() {
		VectorFont.loadFont("unknownfile.ttf");
	}

	/**
	 * Test that loading an ttf that does not exist in the classpath results in the correct error code.
	 */
	@Test
	public void testNonExistingFileErrorCode() {
		int errorCode = Integer.MAX_VALUE;

		try {
			VectorFont.loadFont("unknownfile.ttf");
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}

		Assert.assertEquals("non existing file error code", errorCode, VectorGraphicsException.FONT_INVALID_PATH);
	}

	/**
	 * Test that loading an invalid ttf font throws the correct exception.
	 */
	@Test(expected = VectorGraphicsException.class)
	public void testInvalidTTFFileException() {
		VectorFont.loadFont("/fonts/notattf.txt");
	}

	/**
	 * Test that loading an invalid ttf font results in the correct error code.
	 */
	@Test
	public void testInvalidTTFErrorCode() {
		int errorCode = Integer.MAX_VALUE;

		try {
			VectorFont.loadFont("/fonts/notattf.txt");
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}

		Assert.assertEquals("invalid ttf error code", VectorGraphicsException.FONT_INVALID, errorCode);
	}

	// /**
	// * Test that a valid font can be loaded.
	// */
	// @Test
	// public void testLoadingValidFontCanBeLoaded() {
	// boolean canLoadFont = VectorFont.canLoadFont("/fonts/firstfont.ttf");
	// Assert.assertTrue("can load valid font", canLoadFont);
	// }
	//
	// /**
	// * Test that an invalid font cannot be loaded.
	// */
	// @Test
	// public void testLoadingInvalidFontCannotBeLoaded() {
	// boolean canLoadFont = VectorFont.canLoadFont("/fonts/notattf.txt");
	// Assert.assertFalse("cannot load invalid font", canLoadFont);
	// }
	//
	// /**
	// * Test that an invalid font cannot be loaded.
	// */
	// @Test
	// public void testLoadingNonExistingFontCannotBeLoaded() {
	// boolean canLoadFont = VectorFont.canLoadFont("/fonts/unknownfile.ttf");
	// Assert.assertFalse("cannot load non exsting font", canLoadFont);
	// }

	/**
	 * Test that close() closes the font.
	 */
	@Test
	public void testLoadingCloseFont() {
		VectorFont font = loadTestFont();
		Assert.assertFalse("before close font", font.isClosed());
		font.close();
		Assert.assertTrue("after close font", font.isClosed());
		font.close();
		Assert.assertTrue("second close does nothing", font.isClosed());
	}

	/**
	 * Test that a font can be loaded/closed multiple times.
	 */
	@Test
	public void testLoadingSuccessiveLoadCloseFont() {
		VectorFont font = loadTestFont();
		Assert.assertFalse("before close font", font.isClosed());
		font.close();
		Assert.assertTrue("after close font", font.isClosed());
		font = loadTestFont();
		Assert.assertFalse("before close font", font.isClosed());
		font.close();
		Assert.assertTrue("after close font", font.isClosed());
	}

	/**
	 * Test that first font can be closed.
	 */
	@Test
	public void testRemovingFirstFont() {
		VectorFont font0 = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont font1 = VectorFont.loadFont("/fonts/secondfont.ttf");
		VectorFont font2 = VectorFont.loadFont("/fonts/neuzeit.ttf");

		float f0Width = font0.measureStringWidth("A", 30);
		float f1Width = font1.measureStringWidth("A", 30);
		float f2Width = font2.measureStringWidth("A", 30);

		// ensure widths are different for the next tests
		Assert.assertTrue("width f0f1", f0Width != f1Width);
		Assert.assertTrue("width f0f2", f0Width != f2Width);
		Assert.assertTrue("width f1f2", f1Width != f2Width);

		// check only first font is closed
		font0.close();
		Assert.assertTrue("font0", font0.isClosed());
		Assert.assertFalse("font1", font1.isClosed());
		Assert.assertFalse("font2", font2.isClosed());

		// check the sizes returned by other fonts are unchanged
		Assert.assertEquals("f1Width", f1Width, font1.measureStringWidth("A", 30), 0);
		Assert.assertEquals("f2Width", f2Width, font2.measureStringWidth("A", 30), 0);

		// close all fonts
		font0.close();
		font1.close();
		font2.close();
		Assert.assertTrue("font0", font0.isClosed());
		Assert.assertTrue("font1", font1.isClosed());
		Assert.assertTrue("font2", font2.isClosed());
	}

	/**
	 * Test that second font can be closed.
	 */
	@Test
	public void testRemovingSecondFont() {
		VectorFont font0 = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont font1 = VectorFont.loadFont("/fonts/secondfont.ttf");
		VectorFont font2 = VectorFont.loadFont("/fonts/neuzeit.ttf");

		float f0Width = font0.measureStringWidth("A", 30);
		float f1Width = font1.measureStringWidth("A", 30);
		float f2Width = font2.measureStringWidth("A", 30);

		// ensure widths are different for the next tests
		Assert.assertTrue("width f0f1", f0Width != f1Width);
		Assert.assertTrue("width f0f2", f0Width != f2Width);
		Assert.assertTrue("width f1f2", f1Width != f2Width);

		// check only second font is closed
		font1.close();
		Assert.assertFalse("font0", font0.isClosed());
		Assert.assertTrue("font1", font1.isClosed());
		Assert.assertFalse("font2", font2.isClosed());

		// check the sizes returned by other fonts are unchanged
		Assert.assertEquals("f0Width", f0Width, font0.measureStringWidth("A", 30), 0);
		Assert.assertEquals("f2Width", f2Width, font2.measureStringWidth("A", 30), 0);

		// close all fonts
		font0.close();
		font1.close();
		font2.close();
		Assert.assertTrue("font0", font0.isClosed());
		Assert.assertTrue("font1", font1.isClosed());
		Assert.assertTrue("font2", font2.isClosed());
	}

	/**
	 * Test that last font can be closed.
	 */
	@Test
	public void testRemovingLastFont() {
		VectorFont font0 = VectorFont.loadFont("/fonts/firstfont.ttf");
		VectorFont font1 = VectorFont.loadFont("/fonts/secondfont.ttf");
		VectorFont font2 = VectorFont.loadFont("/fonts/neuzeit.ttf");

		float f0Width = font0.measureStringWidth("A", 30);
		float f1Width = font1.measureStringWidth("A", 30);
		float f2Width = font2.measureStringWidth("A", 30);

		// ensure widths are different for the next tests
		Assert.assertTrue("width f0f1", f0Width != f1Width);
		Assert.assertTrue("width f0f2", f0Width != f2Width);
		Assert.assertTrue("width f1f2", f1Width != f2Width);

		// check only last font is closed
		font2.close();
		Assert.assertFalse("font0", font0.isClosed());
		Assert.assertFalse("font1", font1.isClosed());
		Assert.assertTrue("font2", font2.isClosed());

		// check the sizes returned by other fonts are unchanged
		Assert.assertEquals("f0Width", f0Width, font0.measureStringWidth("A", 30), 0);
		Assert.assertEquals("f1Width", f1Width, font1.measureStringWidth("A", 30), 0);

		// close all fonts
		font0.close();
		font1.close();
		font2.close();
		Assert.assertTrue("font0", font0.isClosed());
		Assert.assertTrue("font1", font1.isClosed());
		Assert.assertTrue("font2", font2.isClosed());
	}

	// /**
	// * Test the loading of a big font file. Note: requires architecture >= 7.17.0 for S3.
	// */
	// @Test(expected = None.class)
	// public void testLoadBigFont() {
	// VectorFont.loadFont("/fonts/bigfont.ttf");
	// }

	private VectorFont loadTestFont() {
		return VectorFont.loadFont("/fonts/firstfont.ttf");
	}
}
