/*
 * Copyright (c) 1997, 2019, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2023 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package javax.crypto;

import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;

/**
 * This class represents a factory for secret keys.
 *
 * <P>
 * Key factories are used to convert <I>keys</I> (opaque cryptographic keys of type {@code Key}) into <I>key
 * specifications</I> (transparent representations of the underlying key material), and vice versa. Secret key factories
 * operate only on secret (symmetric) keys.
 *
 * <P>
 * Key factories are bi-directional, i.e., they allow to build an opaque key object from a given key specification (key
 * material), or to retrieve the underlying key material of a key object in a suitable format.
 *
 * <p>
 * Every implementation of the Java platform is required to support the following standard {@code SecretKeyFactory}
 * algorithms:
 * <ul>
 * <li>{@code DES}</li>
 * <li>{@code DESede}</li>
 * </ul>
 * These algorithms are described in the <a href=
 * "https://docs.oracle.com/en/java/javase/19/docs/specs/security/standard-names.html#secretkeyfactory-algorithms">
 * Signature section</a> of the Java Security Standard Algorithm Names Specification. Consult the release documentation
 * for your implementation to see if any other algorithms are supported.
 *
 * @author Jan Luehe
 *
 * @see SecretKey
 * @see javax.crypto.spec.PBEKeySpec
 * @since 1.4
 */
public class SecretKeyFactory {

	private SecretKeyFactory(String algorithm) throws NoSuchAlgorithmException {
		throw new RuntimeException();
	}

	/**
	 * Returns a {@code SecretKeyFactory} object that converts secret keys of the specified algorithm.
	 *
	 * <p>
	 * This method traverses the list of registered security Providers, starting with the most preferred Provider. A new
	 * SecretKeyFactory object encapsulating the SecretKeyFactorySpi implementation from the first Provider that
	 * supports the specified algorithm is returned.
	 *
	 * <p>
	 *
	 * @param algorithm
	 *            the standard name of the requested secret-key algorithm. See the SecretKeyFactory section in the
	 *            <a href=
	 *            "https://docs.oracle.com/en/java/javase/19/docs/specs/security/standard-names.html#secretkeyfactory-algorithms">
	 *            Signature section</a> of the Java Security Standard Algorithm Names Specification. Consult the release
	 *            documentation for your implementation to see if any other algorithms are supported.
	 *
	 * @return the new {@code SecretKeyFactory} object.
	 *
	 * @exception NullPointerException
	 *                if the specified algorithm is null.
	 *
	 * @exception NoSuchAlgorithmException
	 *                if no Provider supports a SecretKeyFactorySpi implementation for the specified algorithm.
	 *
	 */
	public static final SecretKeyFactory getInstance(String algorithm) throws NoSuchAlgorithmException {
		throw new RuntimeException();
	}

	/**
	 * Returns the algorithm name of this {@code SecretKeyFactory} object.
	 *
	 * <p>
	 * This is the same name that was specified in one of the {@code getInstance} calls that created this
	 * {@code SecretKeyFactory} object.
	 *
	 * @return the algorithm name of this {@code SecretKeyFactory} object.
	 */
	public final String getAlgorithm() {
		throw new RuntimeException();
	}

	/**
	 * Generates a {@code SecretKey} object from the provided key specification (key material).
	 *
	 * @param keySpec
	 *            the specification (key material) of the secret key
	 *
	 * @return the secret key
	 *
	 * @exception InvalidKeySpecException
	 *                if the given key specification is inappropriate for this secret-key factory to produce a secret
	 *                key.
	 */
	public final SecretKey generateSecret(KeySpec keySpec) throws InvalidKeySpecException {
		throw new RuntimeException();
	}

}
