/*
 * Java
 *
 * Copyright 2019-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.net;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.Socket;

import android.net.NetworkCapabilities;
import ej.annotation.Nullable;
import ej.bon.Timer;

/**
 * Basic full java implementation. Consider the network is connected when there is at least one {@link NetworkInterface}
 * which address is not <code>0.0.0.0</code>. It uses an HTTP request to check if it is connected to internet.
 */
public class HttpPollerConnectivityManager extends PollerConnectivityManager {

	/**
	 * Default url used for testing internet connectivity.
	 */
	private static final String HTTP = BASE_PROPERTY + "http."; //$NON-NLS-1$
	/**
	 * Property used to define the port.
	 */
	public static final String PROPERTY_PORT = HTTP + "port"; //$NON-NLS-1$
	/**
	 * Property used to define the host.
	 */
	public static final String PROPERTY_HOST = HTTP + "host"; //$NON-NLS-1$

	/**
	 * Property used to define the payload.
	 */
	public static final String PROPERTY_PAYLOAD = HTTP + "payload"; //$NON-NLS-1$
	/** package for test. **/
	@Nullable
	static final String DEFAULT_PAYLOAD = System.getProperty(PROPERTY_PAYLOAD);
	/** package for test. **/
	static final int DEFAULT_PORT = Integer.getInteger(PROPERTY_PORT, 80).intValue();

	@Nullable
	private static final String DEFAULT_HOSTNAME = System.getProperty(PROPERTY_HOST);
	private static final int READ_BUFFER_SIZE = 1024;
	private static final String HOST = "Host: "; //$NON-NLS-1$
	private static final String GET_HTTP_1_0 = "GET /  HTTP/1.0"; //$NON-NLS-1$
	private static final String NEW_LINE = "\r\n"; //$NON-NLS-1$

	private String payload;
	private String host;
	private int port;

	/**
	 * Instantiates an {@link HttpPollerConnectivityManager} with System properties values.
	 *
	 * It uses the system properties:
	 * <ul>
	 * <li>connectivity.http.host</li>
	 * <li>connectivity.http.port (optional default value is 80)</li>
	 * <li>connectivity.http.payload (optional default value will do a get at the root of the host)</li>
	 * </ul>
	 *
	 * @see PollerConnectivityManager#PollerConnectivityManager()
	 */
	@SuppressWarnings("null")
	public HttpPollerConnectivityManager() {
		super();
		setHttpValues(DEFAULT_HOSTNAME, DEFAULT_PAYLOAD, DEFAULT_PORT);
	}

	/**
	 * Instantiates an {@link HttpPollerConnectivityManager} with System properties values.
	 *
	 * It uses the system properties:
	 * <ul>
	 * <li>connectivity.http.host</li>
	 * <li>connectivity.http.port (optional default value is 80)</li>
	 * <li>connectivity.http.payload (optional default value will do a get at the root of the host)</li>
	 * </ul>
	 *
	 * @param timer
	 *            the timer to use, cannot be <code>null</code>.
	 * @see PollerConnectivityManager#PollerConnectivityManager()
	 */
	@SuppressWarnings("null")
	public HttpPollerConnectivityManager(Timer timer) {
		super(timer);
		setHttpValues(DEFAULT_HOSTNAME, DEFAULT_PAYLOAD, DEFAULT_PORT);
	}

	/**
	 * Instantiates an {@link HttpPollerConnectivityManager} with arguments values.
	 *
	 * @param timer
	 *            the timer to use, cannot be <code>null</code>.
	 * @param delay
	 *            the delay before the start.
	 * @param periodWhenAvaialble
	 *            the period for the polling when at least one interface is available.
	 * @param periodWhenLost
	 *            the period for the polling when all the interface are down.
	 * @param periodWhenInternet
	 *            the period for the polling when there is internet access.
	 */
	@SuppressWarnings("null")
	public HttpPollerConnectivityManager(Timer timer, long delay, long periodWhenAvaialble, long periodWhenLost,
			long periodWhenInternet) {
		super(timer, delay, periodWhenAvaialble, periodWhenLost, periodWhenInternet);
		setHttpValues(DEFAULT_HOSTNAME, DEFAULT_PAYLOAD, DEFAULT_PORT);
	}

	/**
	 * Sets the http values.
	 *
	 * @param host
	 *            the host to use, cannot be <code>null</code>.
	 * @param payload
	 *            the payload to send, if <code>null</code> a simple get request at the root will be used.
	 * @param port
	 *            the port to use.
	 */
	public void setHttpValues(@Nullable String host, @Nullable String payload, int port) {
		if (host == null) {
			throw new IllegalArgumentException();
		}
		this.host = host;
		if (payload != null) {
			this.payload = payload;
		} else {
			this.payload = GET_HTTP_1_0 + NEW_LINE + HOST + host + NEW_LINE + NEW_LINE;
		}
		this.port = port;
	}

	@Override
	protected NetworkCapabilities getCapabilities(InetAddress connectedAddress) {
		NetworkCapabilities capabilities = INTERNET_UNCONNECTED;
		try (Socket socket = new Socket(InetAddress.getByName(this.host), this.port, connectedAddress, 0)) {
			try (OutputStream out = socket.getOutputStream()) {
				out.write(this.payload.getBytes());
				out.flush();

				try (InputStream in = socket.getInputStream()) {
					byte[] b = new byte[READ_BUFFER_SIZE];
					int read;
					do {
						read = in.read(b);
					} while (read >= 0);
				}
				capabilities = INTERNET_CONNECTED;
			}
		} catch (IOException e) {
			log(ConnectivityMessages.ERROR_UNKNOWN, e);
		}
		return capabilities;
	}

}
