/*
 * Java
 *
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.display;

import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;

import ej.microui.display.LLUIDisplay;
import ej.microui.display.MicroUIGraphicsContext;
import ej.microui.display.MicroUIImageFormat;
import ej.microvg.LLVGEngine;
import ej.microvg.MatrixHelper;
import ej.microvg.VGDrawing;
import ej.microvg.paint.TransformationVisitor;
import ej.microvg.paint.VGPaint;

/**
 * VG drawer that targets the BufferedImage with the same format than the display (and the display itself).
 */
public class DisplayDrawer implements VGDrawing {

	@Override
	public MicroUIImageFormat handledFormat() {
		return LLUIDisplay.Instance.getDisplayFormat();
	}

	@Override
	public void drawPath(MicroUIGraphicsContext gc, GeneralPath path, float x, float y, float[] matrix, int fillRule,
			int blend, VGPaint paint) {

		LLVGEngine engine = LLVGEngine.Instance;

		// use a copy of the path to not modify original path
		path = (GeneralPath) path.clone();
		path.setWindingRule(engine.getFillRule(fillRule));

		// Add initial x,y translation on path's matrix
		AffineTransform translatedTransform = new AffineTransform();
		translatedTransform.translate(x, y);
		translatedTransform.concatenate(MatrixHelper.getAffineTransform(matrix, 0, 0));

		// Add initial x,y translation on gradient's matrix
		AffineTransform gradientMatrix = new AffineTransform();
		gradientMatrix.translate(x, y);

		BufferedImage bi = (BufferedImage) gc.getImage().getRAWImage();
		Graphics2D g2d = bi.createGraphics();

		// Set the clip
		engine.setClip(g2d, gc, path, translatedTransform);

		// translate the path
		path.transform(translatedTransform);

		// draw it
		RenderingHints rh = new RenderingHints(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		g2d.setPaint(paint.apply(new TransformationVisitor(gradientMatrix)).getPaint());
		g2d.setRenderingHints(rh);
		g2d.setComposite(engine.getComposite(blend));
		fill(g2d, path);

		// convert colors in drawn area
		final Rectangle2D bounds = g2d.getClipBounds();
		engine.convertColorToColorToDraw(gc, bounds);

	}

	private static void fill(Graphics2D g2d, GeneralPath path) {
		try {
			g2d.fill(path);
		} catch (Throwable e) {
			// unexpected exception: silently ignore it
		}
	}

}
