/*
 * Java
 *
 * Copyright 2009-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.MouseListener;
import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.fp.util.WidgetWithListener;
import ej.microui.event.EventCommand;

/**
 * Widget to simulate a state machine.
 * <p>
 * Notes:
 * <ul>
 * <li>This widget requires a listener: an implementation of {@link WheelListener} to send the state events.</li>
 * </ul>
 */
@SuppressWarnings("nls")
@WidgetDescription(attributes = { @WidgetAttribute(name = "label", isOptional = true), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "maxAngle"), @WidgetAttribute(name = "steps"),
		@WidgetAttribute(name = "pushedSkin"), @WidgetAttribute(name = "filter", isOptional = true),
		@WidgetAttribute(name = "listenerClass", isOptional = true) })
public class Wheel extends WidgetWithListener implements MouseListener {

	/**
	 * Interface that handle wheel events.
	 */
	public static interface WheelListener {

		/**
		 * The specified button has been pressed.
		 *
		 * @param widget
		 *            the turned wheel.
		 *
		 */
		void press(Wheel widget);

		/**
		 * The specified button has been released.
		 *
		 * @param widget
		 *            the turned wheel.
		 *
		 */
		void release(Wheel widget);

		/**
		 * The specified wheel has been turned forward.
		 *
		 * @param widget
		 *            the turned wheel.
		 * @param step
		 *            the new step of the wheel.
		 */
		void turnForward(Wheel widget, int step);

		/**
		 * The specified wheel has been turned backward.
		 *
		 * @param widget
		 *            the turned wheel.
		 * @param step
		 *            the new step of the wheel.
		 */
		void turnBackward(Wheel widget, int step);
	}

	/**
	 * Default implementation of {@link WheelListener}.
	 * <p>
	 * This implementation sends some MicroUI Command events. It targets the common MicroUI Command generator identified
	 * by the tag {@link EventCommand#COMMON_MICROUI_GENERATOR_TAG}.
	 * <p>
	 * This tag can be changed overriding the method {@link #getMicroUIGeneratorTag()}.
	 */
	public static class WheelListenerToCommandEvents implements WheelListener {

		@Override
		public void turnForward(Wheel widget, int step) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.CLOCKWISE);
		}

		@Override
		public void turnBackward(Wheel widget, int step) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.ANTICLOCKWISE);
		}

		@Override
		public void press(Wheel widget) {
			EventCommand.sendEvent(getMicroUIGeneratorTag(), EventCommand.SELECT);
		}

		@Override
		public void release(Wheel widget) {
			// no release event
		}

		/**
		 * Gets the MicroUI Command events generator tag. This generator has to match the generator set during the
		 * MicroEJ platform build in <code>microui/microui.xml</code>.
		 *
		 * @return a MicroUI Command events generator tag.
		 */
		protected String getMicroUIGeneratorTag() {
			return EventCommand.COMMON_MICROUI_GENERATOR_TAG;
		}
	}

	private WheelListener listener;
	private Image[] steps;
	private Image pushedSkin;
	private int xCenter;
	private int yCenter;
	private double lastAngle;
	private double maxAngle;
	private double stepAngle;
	private int current;
	private boolean unending;

	/**
	 * Sets the skin to show the wheel <i>press</i> event.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param pushedSkin
	 *            the <i>press</i> skin.
	 */
	public void setPushedSkin(Image pushedSkin) {
		this.pushedSkin = pushedSkin;
	}

	/**
	 * Sets the bigger angle the wheel can have. A zero-angle means infinite.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param maxAngle
	 *            an angle in degree.
	 */
	public void setMaxAngle(double maxAngle) {
		if (maxAngle < 0) {
			throw new IllegalArgumentException("Angle cannot be negative.");
		}
		this.maxAngle = maxAngle * Math.PI / 180;
	}

	/**
	 * Sets the images which simulate the wheel rotation. The number of images is not defined but first image is for the
	 * wheel state 0° and the last image for the wheel state <i>max</i>°.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param steps
	 *            array of images.
	 */
	public void setSteps(Image[] steps) {
		this.steps = steps;
		setSkin(steps[0]);
	}

	/**
	 * Defines the user class which has to implement {@link WheelListener}.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param listenerClassName
	 *            user listener class name.
	 */
	public void setListenerClass(String listenerClassName) {
		setListenerClass(WheelListener.class, listenerClassName);
	}

	@Override
	public synchronized void finalizeConfiguration() {
		super.finalizeConfiguration();

		this.current = 0;
		this.xCenter = getWidth() >> 1;
		this.yCenter = getHeight() >> 1;
		this.unending = this.maxAngle == 0;

		if (this.unending) {
			this.stepAngle = 360 * Math.PI / 180 / this.steps.length;
		} else {
			this.stepAngle = this.maxAngle / this.steps.length;
		}
	}

	@Override
	public void start() {
		super.start();
		this.listener = newListener(WheelListener.class);
	}

	@Override
	protected Object newDefaultListener() {
		return new WheelListenerToCommandEvents();
	}

	@Override
	public void mousePressed(int x, int y, MouseButton button) {
		// check the mouse button is the right button
		if (button == MouseButton.THIRD_BUTTON) {
			double opp = this.yCenter - y;
			double adj = this.xCenter - x;
			this.lastAngle = Math.atan2(opp, adj);
			press();
		}
	}

	@Override
	public void mouseDragged(int x, int y) {

		// compute angle
		double opp = this.yCenter - y;
		double adj = this.xCenter - x;
		double angle = Math.atan2(opp, adj);

		// compute move
		double shift = this.lastAngle - angle;
		if (shift > Math.PI) {
			shift = -2 * Math.PI + shift;
		}
		if (shift < -Math.PI) {
			shift = 2 * Math.PI - shift;
		}
		if (shift > this.stepAngle) {
			if (turnBackward()) {
				this.lastAngle = angle;
			}
		} else if (shift < -this.stepAngle && turnForward()) {
			this.lastAngle = angle;
		}
	}

	@Override
	public void mouseReleased(int x, int y, MouseButton button) {
		// check the mouse button is the right.
		if (button == MouseButton.THIRD_BUTTON) {
			release();
		}
	}

	private boolean turnForward() {
		int old = this.current;
		int length = this.steps.length;
		this.current = old + 1;
		if (this.current >= length - 1) {
			if (this.unending) {
				this.current %= length;
			} else {
				this.current = length - 1;
			}
		}
		if (old != this.current) {
			setCurrentSkin(this.steps[this.current]);
			this.listener.turnForward(this, this.current);
			return true;
		}
		return false;
	}

	private boolean turnBackward() {
		int old = this.current;
		int length = this.steps.length;
		this.current = old - 1;
		if (this.current < 0) {
			if (this.unending) {
				this.current = (this.current % length) + length;
			} else {
				this.current = 0;
			}
		}
		if (old != this.current) {
			setCurrentSkin(this.steps[this.current]);
			this.listener.turnBackward(this, this.current);
			return true;
		}
		return false;
	}

	private void press() {
		setCurrentSkin(this.pushedSkin);
		this.listener.press(this);
	}

	private void release() {
		setCurrentSkin(this.steps[this.current]);
		this.listener.release(this);
	}

	@Override
	public void dispose() {
		FrontPanel.getFrontPanel().disposeIfNotNull(this.steps);
		this.steps = null;
		super.dispose();
	}
}
