/*
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector;

import ej.annotation.Nullable;
import ej.mwt.Widget;

/**
 * A state selector selects from a state. That means that the state of this selector is available in the widget states.
 * <p>
 * Equivalent to pseudo-class (eg. <code>:active</code>) selector in CSS. Its specificity is (0,0,1,0).
 *
 * @see Widget#isInState(int)
 * @see SelectorHelper
 */
public class StateSelector implements Selector {

	private final int state;

	/**
	 * Creates a state selector with one state.
	 *
	 * @param state
	 *            the state.
	 */
	public StateSelector(int state) {
		this.state = state;
	}

	@Override
	public boolean appliesToWidget(Widget widget) {
		// Check that the state of this selector is available in the given widget.
		return widget.isInState(this.state);
	}

	@Override
	public int getSpecificity() {
		return SelectorHelper.getSpecificity(0, 0, 1, 0);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && getClass() == obj.getClass()) {
			StateSelector other = (StateSelector) obj;
			return this.state == other.state;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return this.state;
	}
}
