/*
 * Copyright (c) 2003, 2013, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package sun.misc;

import java.util.Arrays;

public class FormattedFloatingDecimal {

	public static FormattedFloatingDecimal valueOf(double d, int precision, Object arg) {
		FloatingDecimal.BinaryToASCIIConverter fdConverter = FloatingDecimal.getBinaryToASCIIConverter(d, arg);
		return new FormattedFloatingDecimal(precision, fdConverter);
	}

	private char[] mantissa;

	private FormattedFloatingDecimal(int precision, FloatingDecimal.BinaryToASCIIConverter fdConverter) {
		if (fdConverter.isExceptional()) {
			this.mantissa = fdConverter.toJavaFormatString().toCharArray();
			return;
		}
		char[] digits = new char[20];
		int nDigits = fdConverter.getDigits(digits);
		int decExp = fdConverter.getDecimalExponent();
		int exp;
		boolean isNegative = fdConverter.isNegative();
		exp = applyPrecision(decExp, digits, nDigits, decExp + precision);
		fillDecimal(precision, digits, nDigits, exp, isNegative);
	}

	public char[] getMantissa() {
		return this.mantissa;
	}

	/**
	 * Returns new decExp in case of overflow.
	 */
	private static int applyPrecision(int decExp, char[] digits, int nDigits, int prec) {
		if (prec >= nDigits || prec < 0) {
			// no rounding necessary
			return decExp;
		}
		if (prec == 0) {
			// only one digit (0 or 1) is returned because the precision
			// excludes all significant digits
			if (digits[0] >= '5') {
				digits[0] = '1';
				Arrays.fill(digits, 1, nDigits, '0');
				return decExp + 1;
			} else {
				Arrays.fill(digits, 0, nDigits, '0');
				return decExp;
			}
		}
		int q = digits[prec];
		if (q >= '5') {
			int i = prec;
			q = digits[--i];
			if (q == '9') {
				while (q == '9' && i > 0) {
					q = digits[--i];
				}
				if (q == '9') {
					// carryout! High-order 1, rest 0s, larger exp.
					digits[0] = '1';
					Arrays.fill(digits, 1, nDigits, '0');
					return decExp + 1;
				}
			}
			digits[i] = (char) (q + 1);
			Arrays.fill(digits, i + 1, nDigits, '0');
		} else {
			Arrays.fill(digits, prec, nDigits, '0');
		}
		return decExp;
	}

	private static char[] create(boolean isNegative, int size) {
		if (isNegative) {
			char[] r = new char[size + 1];
			r[0] = '-';
			return r;
		} else {
			return new char[size];
		}
	}

	/*
	 * Fills mantissa char arrays for DECIMAL_FLOAT format. Exponent should be equal to null.
	 */
	private void fillDecimal(int precision, char[] digits, int nDigits, int exp, boolean isNegative) {
		int startIndex = isNegative ? 1 : 0;
		if (exp > 0) {
			// print digits.digits.
			if (nDigits < exp) {
				this.mantissa = create(isNegative, exp);
				System.arraycopy(digits, 0, this.mantissa, startIndex, nDigits);
				Arrays.fill(this.mantissa, startIndex + nDigits, startIndex + exp, '0');
				// Do not append ".0" for formatted floats since the user
				// may request that it be omitted. It is added as necessary
				// by the Formatter.
			} else {
				int t = Math.min(nDigits - exp, precision);
				this.mantissa = create(isNegative, exp + (t > 0 ? (t + 1) : 0));
				System.arraycopy(digits, 0, this.mantissa, startIndex, exp);
				// Do not append ".0" for formatted floats since the user
				// may request that it be omitted. It is added as necessary
				// by the Formatter.
				if (t > 0) {
					this.mantissa[startIndex + exp] = '.';
					System.arraycopy(digits, exp, this.mantissa, startIndex + exp + 1, t);
				}
			}
		} else if (exp <= 0) {
			int zeros = Math.max(0, Math.min(-exp, precision));
			int t = Math.max(0, Math.min(nDigits, precision + exp));
			// write '0' s before the significant digits
			if (zeros > 0) {
				this.mantissa = create(isNegative, zeros + 2 + t);
				this.mantissa[startIndex] = '0';
				this.mantissa[startIndex + 1] = '.';
				Arrays.fill(this.mantissa, startIndex + 2, startIndex + 2 + zeros, '0');
				if (t > 0) {
					// copy only when significant digits are within the precision
					System.arraycopy(digits, 0, this.mantissa, startIndex + 2 + zeros, t);
				}
			} else if (t > 0) {
				this.mantissa = create(isNegative, zeros + 2 + t);
				this.mantissa[startIndex] = '0';
				this.mantissa[startIndex + 1] = '.';
				// copy only when significant digits are within the precision
				System.arraycopy(digits, 0, this.mantissa, startIndex + 2, t);
			} else {
				this.mantissa = create(isNegative, 1);
				this.mantissa[startIndex] = '0';
			}
		}
	}

}
