/*
 * Copyright 2010-2015 http://beders.github.com/Resty
 * Copyright 2015-2019 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ej.rest.web;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.logging.Logger;

import org.json.me.JSONException;
import org.json.me.JSONObject;

import ej.jsonpath.parser.CharStream;
import ej.jsonpath.parser.JSONPathCompiler;
import ej.jsonpath.parser.ParseException;
import ej.jsonpath.parser.SimpleCharStream;

/**
 * Create a path query for a JSON object. Usualy this is done by calling Resty.path(...)
 *
 * A JSONPathQuery is similar to JsonPath or XPath in that you can create access paths into an JSON object. It uses dot
 * '.' as a path separator and [] to access arrays. The array index is either a number or an expression to evaluate for
 * each object in the array. The first matching object is returned. I.e.
 * <p>
 * <code>store.book[price&gt;7 &amp;&amp; price&lt;12.999].author</code>
 * <p>
 * In the example above the JSON object with a prive value between 7 and 12.999 is selected and the author returned.
 * Boolean expressions are supported with &amp;&amp; (and), || (or), ! (not). Values can be compared with =,&lt;,&gt;.
 *
 *
 * @author beders
 *
 */
public class JSONPathQuery extends PathQuery<JSONResource, Object> {
	private JSONPathCompiler compiler;
	private final String expr;
	static private Logger log = Logger.getLogger(JSONPathQuery.class.getName());

	public JSONPathQuery(String anExpression) {
		this.expr = anExpression;
	}

	@Override
	Object eval(JSONResource resource) throws JSONException, ParseException, IOException {
		JSONObject json = resource.object();
		log.fine("JSON Received:" + json);
		Object result = getCompiler().json().eval(json);
		return result;
	}

	protected JSONPathCompiler getCompiler() throws ParseException {
		if (null == this.compiler) {
			CharStream charStream = new SimpleCharStream(new ByteArrayInputStream(this.expr.getBytes()));
			this.compiler = new JSONPathCompiler(charStream);
		}
		return this.compiler;
	}

}
